package ru.yandex.chemodan.app.docviewer.copy.resourcemanagers;

import java.io.IOException;
import java.net.URI;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ResponseHandler;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.StorageResourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.storage.StorageId;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

@AllArgsConstructor
public class YaDiskCopierResponseHandlerVersioning implements ResponseHandler<StorageResourceInfo> {

    private static final Logger logger = LoggerFactory.getLogger(YaDiskCopierResponseHandlerVersioning.class);

    private static final ObjectMapper mapper = new ObjectMapper();
    private final URI uri;
    private final Instant startTime;

    @Override
    public StorageResourceInfo handleResponse(HttpResponse response) throws  IOException {
        LoggerEventsRecorder.saveGetMulcaIdEvent(
                uri, response.getStatusLine().toString(), TimeUtils.toDurationToNow(startTime));

        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode == HttpStatus.SC_NOT_FOUND) {
            throw new UserException(ErrorCode.FILE_NOT_FOUND);
        } else if (statusCode == HttpStatus.SC_FORBIDDEN) {
            throw new UserException(ErrorCode.FILE_IS_FORBIDDEN);
        } else if (statusCode != HttpStatus.SC_OK) {
            throw new UserException(ErrorCode.UNABLE_TO_RETRIEVE_FILE,
                    "No redirect to mulca from ya-disk, status-code: " + statusCode);
        }

        String content = InputStreamSourceUtils.wrap(response.getEntity().getContent()).readText("utf-8");
        return parseInfo(content);
    }

    /**
     * {
     *    "name": <file_name>,
     *    "file_stid": <stid>,
     *    "mimetype": text/plain
     *    "media_type": document
     * }
     */
    private StorageResourceInfo parseInfo(String content) {
        JsonNode root = readTree(content);
        logger.debug("fetched content {}", content);
        String name = root.get("name").textValue();
        String mimetype = root.get("mimetype").textValue();
        StorageId storageId = StorageId.fromMulcaId(MulcaId.fromSerializedString(root.get("file_stid").textValue()));
        return new StorageResourceInfo(storageId, trimQuotes(Option.of(name)), Option.of(mimetype), Option.empty());
    }

    private JsonNode readTree(String response) {
        try {
            return mapper.readTree(response);
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    private static Option<String> trimQuotes(Option<String> stringO) {
        if (!stringO.isPresent()) {
            return Option.empty();
        }
        String s = stringO.get();
        return Option.of(s.length() >= 2 && s.startsWith("\"") && s.endsWith("\"")
                ? s.substring(1, s.length() - 1)
                : s);
    }
}
