package ru.yandex.chemodan.app.docviewer.dao.pdfImage;

import java.util.Date;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.log.DocviewerTskvEvent;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.StoredImage;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author akirakozov
 */
public class MongoImageDao implements ImageDao {
    public static final String COLLECTION = "pdf-images-cache";

    private static final String COLUMN_ID = "_id";
    private static final String SUBCOLUMN_FILE_ID = "file-id";
    private static final String SUBCOLUMN_SUB_ID = "sub-id";

    private static final String COLUMN_LAST_ACCESS = "last_access";
    private static final String COLUMN_LINK = "link";

    @Autowired
    @Qualifier("mongoDbAdapter")
    private MongoDbAdapter mongoDbAdapter;

    @Value("${cleanup.threads}")
    private int cleanupThreads;

    private static StoredImage deserialize(DBObject dbObject) {
        Object id = dbObject.get(COLUMN_ID);
        Validate.isTrue(id instanceof DBObject, "DBObject with ID '" + id + "' has incorrect structure");

        DBObject idFields = (DBObject) id;
        String fileId = String.valueOf(idFields.get(SUBCOLUMN_FILE_ID));
        String subKey = String.valueOf(idFields.get(SUBCOLUMN_SUB_ID));

        Instant lastAccess = MongoDbUtils.getInstant(dbObject, COLUMN_LAST_ACCESS).getOrElse(TimeUtils.now());
        String fileLink = String.valueOf(dbObject.get(COLUMN_LINK));

        return new StoredImage(fileId, subKey, lastAccess, fileLink);
    }

    private static Function<DBObject, StoredImage> deserializeF() {
        return MongoImageDao::deserialize;
    }

    private DBCollection getCollection() {
        return mongoDbAdapter.getDatabase().getCollection(COLLECTION);
    }

    private static DBObject getIdKeyObject(String fileId, String subKey) {
        DBObject id = new BasicDBObject();
        id.put(SUBCOLUMN_FILE_ID, fileId);
        id.put(SUBCOLUMN_SUB_ID, subKey);
        return new BasicDBObject(COLUMN_ID, id);
    }

    private static DBObject getIdKeyObject(String fileId, int oneBasedPageIndex, DimensionO size) {
        return getIdKeyObject(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size));
    }

    private DBObject getFileIdKeyObject(String fileId) {
        return new BasicDBObject(COLUMN_ID + "." + SUBCOLUMN_FILE_ID, fileId);
    }

    @Override
    public Option<StoredImage> findOne(String fileId, int oneBasedPageIndex, DimensionO size) {
        return MongoDbUtils.findOne(getCollection(), getIdKeyObject(fileId, oneBasedPageIndex, size), deserializeF());
    }

    @Override
    public Option<StoredImage> findOne(String fileId, String subKey) {
        return MongoDbUtils.findOne(getCollection(), getIdKeyObject(fileId, subKey), deserializeF());
    }

    @Override
    public void updateLastAccessTime(String fileId, int oneBasedPageIndex, DimensionO size) {
        updateLastAccessTime(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size));
    }

    @Override
    public void updateLastAccessTime(String fileId, String subKey) {
        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_LAST_ACCESS, new Date());
        DBObject update = new BasicDBObject("$set", toUpdate);
        getCollection().update(getIdKeyObject(fileId, subKey), update, false, false);
        DocviewerTskvEvent.imageAccessed(fileId, subKey).log();
    }

    @Override
    public void savePdfImageIfNotExists(String fileId, int oneBasedPageIndex,
            DimensionO size, final String fileLinkString)
    {
        saveImageIfNotExists(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size), fileLinkString);
    }

    @Override
    public void saveImageIfNotExists(String fileId, String subKey, String fileLinkString) {
        DBObject query = getIdKeyObject(fileId, subKey);
        {
            DBObject toUpdate = new BasicDBObject();
            toUpdate.put(COLUMN_LAST_ACCESS, new Date());
            toUpdate.put(COLUMN_LINK, fileLinkString);
            DBObject update = new BasicDBObject("$set", toUpdate);

            getCollection().update(query, update, true, false);
            DocviewerTskvEvent.imageAccessed(fileId, subKey).log();
        }
    }

    @Override
    public void delete(String fileId, int oneBasedPageIndex, DimensionO size) {
        // TODO try-catch, save cleanup event  /  log error, save failed event
        DBObject idKeyObject = getIdKeyObject(fileId, oneBasedPageIndex, size);
        getCollection().remove(idKeyObject);
        DocviewerTskvEvent.imageDeleted(
                Option.ofNullable(idKeyObject.get(SUBCOLUMN_FILE_ID)).map(Object::toString).getOrElse(""),
                Option.ofNullable(idKeyObject.get(SUBCOLUMN_SUB_ID)).map(Object::toString).getOrElse("")).log();
    }

    @Override
    public void delete(String fileId, String subKey) {
        getCollection().remove(getIdKeyObject(fileId, subKey));
        DocviewerTskvEvent.imageDeleted(fileId, subKey).log();
    }

    @Override
    public void deleteByLastAccessLessBatch(Instant timestamp, Function1V<StoredImage> deleteHandler) {
        DBObject query = new BasicDBObject(COLUMN_LAST_ACCESS, new BasicDBObject("$lt", timestamp.toDate()));
        MongoDbUtils.forEachBatch(getCollection(), query,
                COLUMN_LAST_ACCESS, cleanupThreads, deserializeF().andThen(deleteHandler));
    }

    @Override
    public void deleteByFileIdBatch(String fileId, Function1V<StoredImage> deleteHandler) {
        MongoDbUtils.forEachBatch(getCollection(), getFileIdKeyObject(fileId), COLUMN_ID,
                 cleanupThreads, deserializeF().andThen(deleteHandler));
    }
}
