package ru.yandex.chemodan.app.docviewer.dao.results;

import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import com.mongodb.ReadPreference;
import lombok.val;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.PagesInfoHelper;
import ru.yandex.commune.alive2.AliveAppInfo;

public class MongoStoredResultDao implements StoredResultDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoStoredResultDao.class);

    public static final String COLLECTION = "results";

    private static final String COLUMN_ID = "_id";
    private static final String SUBCOLUMN_FILE_ID = "file-id";
    private static final String SUBCOLUMN_CONVERT_TARGET = "convert-target";

    private static final String COLUMN_FILE_LINK = "file-link";
    private static final String COLUMN_LAST_ACCESS_TIME = "last-access-time";
    private static final String COLUMN_LENGTH = "length";
    private static final String COLUMN_PAGES = "pages";

    private static final String COLUMN_PAGES_INFO = "pages-info";

    private static final String WEIGHT = "weight";

    private static final String COLUMN_TYPE = "type";

    private static final String COLUMN_ERROR = "error";
    private static final String COLUMN_ERROR_CODE = "error-code";
    private static final String COLUMN_FAILS_COUNT = "fails-count";
    private static final String COLUMN_VERSION = "version";
    private static final String COLUMN_REMOTE_KEY = "remote-id";

    private static final String COLUMN_PASSWORD_HASH = "password-hash";
    private static final String COLUMN_CONTENT_TYPE = "content-type";

    private static final String COLUMN_RESTORE_URI = "restore-uri";

    private static final String COLUMN_DOCVIEWER_VERSION = "docviewer-version";

    private static final Set<String> KNOWN_COLUMNS = Collections
            .unmodifiableSet(new HashSet<>(Arrays.asList(
                    COLUMN_ID, COLUMN_FILE_LINK, COLUMN_LAST_ACCESS_TIME,
                    COLUMN_LENGTH, COLUMN_PAGES, COLUMN_PAGES_INFO,
                    WEIGHT, COLUMN_TYPE,
                    COLUMN_ERROR, COLUMN_ERROR_CODE,
                    COLUMN_PASSWORD_HASH, COLUMN_CONTENT_TYPE, COLUMN_RESTORE_URI, COLUMN_DOCVIEWER_VERSION)));

    @Autowired
    @Qualifier("mongoDbAdapter")
    private MongoDbAdapter mongoDbAdapter;

    @Value("${cleanup.threads}")
    private int cleanupThreads;

    @Autowired
    private AliveAppInfo aliveAppInfo;

    private DBCollection getCollection() {
        return mongoDbAdapter.getDatabase().getCollection(COLLECTION);
    }

    @Override
    public void delete(String fileId, TargetType convertTargetType) {
        try {
            final DBObject query = getIdKeyObject(fileId, convertTargetType);
            getCollection().remove(query);
            LoggerEventsRecorder.saveCleanupResultEvent(COLLECTION, fileId, Option.of(convertTargetType));
        } catch (Exception exc) {
            logger.error("Unable to delete stored words for file '" + fileId + "': " + exc, exc);
            LoggerEventsRecorder.saveCleanupResultFailedEvent(COLLECTION, exc, fileId, Option.of(convertTargetType));
        }
    }

    private StoredResult deserialize(DBObject dbObject) {
        StoredResult storedResult = new StoredResult();

        Object id = dbObject.get(COLUMN_ID);
        if (!(id instanceof DBObject)) {
            throw new IllegalStateException("DBObject with ID '" + id + "' has incorrect structure");
        }

        DBObject idFields = (DBObject) id;
        storedResult.setFileId(String.valueOf(idFields.get(SUBCOLUMN_FILE_ID)));
        storedResult.setConvertTargetType(MongoDbUtils.getEnum(idFields, SUBCOLUMN_CONVERT_TARGET,
                TargetType.getResolver()).get());

        storedResult.setError(MongoDbUtils.getString(dbObject, COLUMN_ERROR, true, false));
        if (dbObject.containsField(COLUMN_ERROR_CODE)) {
            final String errorCodeStr = String.valueOf(dbObject.get(COLUMN_ERROR_CODE));
            final ErrorCode errorCode = ErrorCode.valueOf(errorCodeStr,
                    ErrorCode.UNKNOWN_CONVERT_ERROR);
            storedResult.setErrorCode(Option.of(errorCode));
        }

        if (dbObject.containsField(COLUMN_FAILS_COUNT)) {
            storedResult.setFailedAttemptsCount(MongoDbUtils.getInteger(dbObject, COLUMN_FAILS_COUNT));
        }

        if (dbObject.containsField(COLUMN_PASSWORD_HASH)) {
            storedResult.setPasswordHash(MongoDbUtils.getStringNoEmpties(dbObject, COLUMN_PASSWORD_HASH));
        }

        if (dbObject.containsField(COLUMN_LAST_ACCESS_TIME)) {
            storedResult.setLastAccess(MongoDbUtils.getInstant(dbObject, COLUMN_LAST_ACCESS_TIME)
                    .getOrThrow(COLUMN_LAST_ACCESS_TIME));
        }
        if (dbObject.containsField(COLUMN_REMOTE_KEY)) {
            storedResult.setRemoteFileId(MongoDbUtils.getStringNoEmpties(dbObject, COLUMN_REMOTE_KEY));
        }
        if (dbObject.containsField(COLUMN_CONTENT_TYPE)) {
            storedResult.setContentType(MongoDbUtils.getStringNoEmpties(dbObject, COLUMN_CONTENT_TYPE));
        }
        if (dbObject.containsField(COLUMN_RESTORE_URI)) {
            storedResult.setRestoreUri(MongoDbUtils.getStringNoEmpties(dbObject, COLUMN_RESTORE_URI));
        }
        if (dbObject.containsField(COLUMN_DOCVIEWER_VERSION)) {
            storedResult.setDocviewerVersion(MongoDbUtils.getStringNoEmpties(dbObject, COLUMN_DOCVIEWER_VERSION));
        }

        storedResult.setFileLink(MongoDbUtils.getString(dbObject, COLUMN_FILE_LINK, true, false));
        storedResult.setLength(MongoDbUtils.getLong(dbObject, COLUMN_LENGTH));
        storedResult.setPages(MongoDbUtils.getInteger(dbObject, COLUMN_PAGES));
        storedResult.setPagesInfo(PagesInfoHelper.load(dbObject.get(COLUMN_PAGES_INFO)));
        storedResult.setWeight(MongoDbUtils.getLong(dbObject, WEIGHT).getOrElse(0L));

        Option<ConvertResultType> type = MongoDbUtils.getEnum(dbObject, COLUMN_TYPE, ConvertResultType.getResolver());

        storedResult.setConvertResultType(type);


        DocumentProperties properties = DocumentProperties.EMPTY;
        for (String key : dbObject.keySet()) {
            if (KNOWN_COLUMNS.contains(key)) {
                continue;
            }
            properties = properties.withProperty(key, String.valueOf(dbObject.get(key)));
        }
        storedResult.setProperties(Option.of(properties));

        storedResult.setPackageVersion(MongoDbUtils.getString(dbObject, COLUMN_VERSION, true, false));

        return storedResult;
    }

    @Override
    public Option<StoredResult> find(String fileId, TargetType convertTargetType) {
        return findOne(fileId, convertTargetType);
    }

    @Override
    public Option<StoredResult> findAnyOfTypes(String fileId, ListF<TargetType> targetTypes) {
        if (targetTypes.isEmpty()) {
            return Option.empty();
        }
        ListF<DBObject> idsToScan = targetTypes.map(type -> getIdSubkeyObject(fileId, type));
        DBObject query = new BasicDBObject(COLUMN_ID, new BasicDBObject("$in", idsToScan));
        return MongoDbUtils.findOneSorted(getCollection(), query, new BasicDBObject(), this::deserialize);
    }

    @Override
    public Iterable<StoredResult> findByLastAccessLess(Instant timestamp) {
        DBObject query = new BasicDBObject(COLUMN_LAST_ACCESS_TIME, new BasicDBObject("$lt", timestamp.toDate()));
        return MongoDbUtils.find(getCollection(), query, this::deserialize);
    }

    @Override
    public void deleteByLastAccessLessBatch(Instant timestamp, Function1V<StoredResult> deleteHandler) {
        DBObject query = new BasicDBObject(COLUMN_LAST_ACCESS_TIME, new BasicDBObject("$lt", timestamp.toDate()));
        MongoDbUtils.forEachBatch(getCollection(), query, COLUMN_LAST_ACCESS_TIME,
                cleanupThreads, ((Function<DBObject, StoredResult>) this::deserialize).andThen(deleteHandler));
    }

    private Option<StoredResult> findOne(String fileId, TargetType convertTargetType) {
        val key = getIdKeyObject(fileId, convertTargetType);
        return MongoDbUtils.findOne(getCollection(), key, this::deserialize)
                .orElse(() -> MongoDbUtils.findOne(getCollection(), key, ReadPreference.primary(), this::deserialize));
    }

    private DBObject getIdKeyObject(String fileId, TargetType convertTargetType) {
        return new BasicDBObject(COLUMN_ID, getIdSubkeyObject(fileId, convertTargetType));
    }

    private DBObject getIdSubkeyObject(String fileId, TargetType convertTargetType) {
        DBObject id = new BasicDBObject();
        id.put(SUBCOLUMN_FILE_ID, fileId);
        id.put(SUBCOLUMN_CONVERT_TARGET, convertTargetType.name());
        return id;
    }

    @Override
    public void saveOrUpdateResult(ConvertErrorArgs args) {

        DBObject query = getIdKeyObject(args.getFileId(), args.getTargetType());

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_ERROR, args.getError());
        toUpdate.put(COLUMN_ERROR_CODE, args.getErrorCode().toString());
        toUpdate.put(COLUMN_LAST_ACCESS_TIME, new Date());
        toUpdate.put(COLUMN_FAILS_COUNT, args.getFailedAttemptsCount());
        toUpdate.put(COLUMN_VERSION, args.getPackageVersion());
        toUpdate.put(COLUMN_DOCVIEWER_VERSION, aliveAppInfo.getVersion());
        final DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, true, false);
    }

    @Override
    public void saveOrUpdateResult(ConvertSuccessArgs args) {
        DBObject query = getIdKeyObject(args.getFileId(), args.getTargetType());

        DBObject update = new BasicDBObject();
        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_LAST_ACCESS_TIME, new Date());
        toUpdate.put(COLUMN_FILE_LINK, args.getResultFileLink());
        toUpdate.put(COLUMN_LENGTH, args.getLength());
        toUpdate.put(COLUMN_PAGES, args.getPages());
        toUpdate.put(WEIGHT, args.getWeight());
        toUpdate.put(COLUMN_RESTORE_URI, args.getRestoreUri());
        toUpdate.put(COLUMN_DOCVIEWER_VERSION, aliveAppInfo.getVersion());
        if (args.getPagesInfo().isPresent()) {
            toUpdate.put(COLUMN_PAGES_INFO, PagesInfoHelper.toDBObject(args.getPagesInfo().get()));
        }
        toUpdate.put(COLUMN_TYPE, args.getType().name());
        if (args.getRawPassword().isPresent()) {
            toUpdate.put(COLUMN_PASSWORD_HASH, SessionKey.toHashValue(args.getRawPassword().get()));
        }
        if (args.getRemoteFileId().isPresent()) {
            toUpdate.put(COLUMN_REMOTE_KEY, args.getRemoteFileId().get());
        }
        if (args.getContentType().isPresent()) {
            toUpdate.put(COLUMN_CONTENT_TYPE, args.getContentType().get());
        }
        toUpdate.putAll(args.getProperties().get());
        update.put("$set", toUpdate);

        getCollection().update(query, update, true, false);
    }

    @Override
    public void addExtractedText(String fileId, TargetType convertTargetType, String link) {
        DBObject query = getIdKeyObject(fileId, convertTargetType);

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(DocumentProperties.EXTRACTED_TEXT, link);
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void setRestoreUri(String fileId, TargetType convertTargetType, String restoreUri) {
        DBObject query = getIdKeyObject(fileId, convertTargetType);

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_RESTORE_URI, restoreUri);
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void updateLastAccessTime(String fileId, TargetType convertTargetType) {
        DBObject query = getIdKeyObject(fileId, convertTargetType);
        query.put(COLUMN_LAST_ACCESS_TIME, new BasicDBObject("$lt", new Date()));

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_LAST_ACCESS_TIME, new Date());
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void updateWeight(String fileId, TargetType convertTargetType, long weight) {
        DBObject query = getIdKeyObject(fileId, convertTargetType);

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(WEIGHT, weight);
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void handleEachResult(Function1V<StoredResult> handler) {
        MongoDbUtils.forEach(getCollection(), ((Function<DBObject, StoredResult>) this::deserialize).andThen(handler));
    }

    @Override
    public ListF<StoredResult> findByRemoteId(String remoteId) {
        return MongoDbUtils.find(getCollection(), new BasicDBObject(COLUMN_REMOTE_KEY, remoteId), this::deserialize);
    }
}
