package ru.yandex.chemodan.app.docviewer.dao.results;

import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author vlsergey
 */
@Data
@BenderBindAllFields
public class StoredResult {
    public static final String RESTORE_URI_IS_EXTERNAL_FIELD = "isExternalUri";

    private static final int COMPACT_ERROR_LINES = 5;

    private String fileId;
    private TargetType convertTargetType;
    private Option<String> error = Option.empty();
    private Option<ErrorCode> errorCode = Option.empty();
    private Option<Integer> failedAttemptsCount = Option.empty();
    private Option<String> passwordHash = Option.empty();
    private Option<String> fileLink = Option.empty();
    private Instant lastAccess = TimeUtils.now();
    private Option<Long> length = Option.empty();
    private Option<Integer> pages = Option.empty();
    private Option<PagesInfo> pagesInfo = Option.empty();
    private Option<ConvertResultType> convertResultType = Option.empty();
    private Option<String> packageVersion = Option.empty();
    private long weight = 0;
    private Option<String> remoteFileId = Option.empty();
    private Option<String> contentType = Option.empty();
    private Option<String> restoreUri = Option.empty();
    private Option<String> docviewerVersion = Option.empty();
    private Option<DocumentProperties> properties = Option.empty();

    public DocumentProperties getDocumentProperties() {
        return properties.getOrElse(DocumentProperties.EMPTY);
    }

    public TargetType getConvertTargetType() {
        return convertTargetType;
    }

    public Option<String> getCompactError() {
        return error.map(a -> {
            ListF<String> lines = StringUtils.lines(a);
            return lines.take(COMPACT_ERROR_LINES).mkString("\n") +
                    (lines.size() > COMPACT_ERROR_LINES ? "\n..." : "");
        });
    }

    public boolean isConvertResultTypePdf() {
        return convertResultType.isSome(ConvertResultType.PDF);
    }

    public boolean isConvertResultTypeZippedHtml() {
        return convertResultType.isSome(ConvertResultType.ZIPPED_HTML) ||
                // XXX temporary migration code: is needed to handle situation:
                // .txt file has been previously converted with HTML_WITH_IMAGES;
                // db contains record without type, and with isPdf = false.
                // So if Html*Action calls this method, none type is ZIPPED_HTML.
                !convertResultType.isPresent();
    }

    // 'width-sensitive' affects total page width and indents. False means fixed big width of white background.
    // True means small padding and exact width (defined by pages info, otherwise is can be changed to default).
    // Besides pdf result type, we need to set true to images (which define pages info, to prevent stretching).
    // Here is no info for detected content family, but for now pages info presents only for pdf and images.
    public boolean isWidthSensitive() {
        return isConvertResultTypePdf() || pagesInfo.isPresent(); // && uri[detected-content-family] == 'image'
    }

    public static String createRestoreUri(ActualUri uri, boolean isExternalUri) {
        return UrlUtils.addParameter(uri.getUriString(), StoredResult.RESTORE_URI_IS_EXTERNAL_FIELD, isExternalUri);
    }

    public void setFileId(String fileId) {
        this.fileId = fileId;
    }
}
