package ru.yandex.chemodan.app.docviewer.dao.rights;

import java.util.Date;
import java.util.Optional;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.BulkWriteOperation;
import com.mongodb.DB;
import com.mongodb.DBObject;
import com.mongodb.ReadPreference;
import lombok.Getter;
import lombok.val;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.commune.mongo.MongoHolder;
import ru.yandex.commune.mongo.bender.MongoId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;

/**
 * @author vlsergey
 * @author ssytnik
 */
public class MongoUriRightsDao implements UriRightsDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoUriRightsDao.class);

    // TODO rename to uri-rights, and, after rights-files removal, to just rights (if not removed itself)
    private static final String COLLECTION = "rights-urls";

    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_URI = "uri";
    private static final String COLUMN_URI_HASH = "uri_hash";
    private static final String COLUMN_UID = "uid";
    private static final String COLUMN_FILE_ID = "file-id";
    private static final String COLUMN_TIMESTAMP = "timestamp";

    private final MongoCollection<String, UriRights> collection;

    @Bendable
    @BenderMembersToBind(MembersToBind.WITH_ANNOTATIONS)
    private static class UriRights {

        @MongoId
        @BenderPart(name = MongoUriRightsDao.COLUMN_ID, strictName = true)
        private String id;

        @Getter
        @BenderPart(name = MongoUriRightsDao.COLUMN_URI, strictName = true)
        private String uri;

        @BenderPart(name = MongoUriRightsDao.COLUMN_URI_HASH, strictName = true)
        private String uriHash;

        @BenderPart(name = MongoUriRightsDao.COLUMN_UID, strictName = true)
        private long uid;

        @BenderPart(name = MongoUriRightsDao.COLUMN_FILE_ID, strictName = true)
        private Optional<String> fileId;

        @Getter
        @BenderPart(name = MongoUriRightsDao.COLUMN_TIMESTAMP, strictName = true)
        private Instant timestamp;

    }

    public MongoUriRightsDao(DB db) {
        collection = new MongoCollection<>(
                db.getCollection(COLLECTION), UriRights.class,
                MongoHolder.defaultBenderConfiguration(),
                MongoDefaultInterceptors.defaultInterceptors());
    }

    // XXX should be consistent to MongoFileRightsDao (for debug): apply md5 to uri, not whole string
    public static String generateId(ActualUri uri, PassportUidOrZero uid) {
        return MongoDbUtils.calculateMd5IdKey("url:" + uid.getUid() + ":" + uri.getUriString());
    }

    public static BasicDBObject getIdKeyObject(ActualUri uri, PassportUidOrZero uid) {
        return new BasicDBObject(COLUMN_ID, generateId(uri, uid));
    }

    @Override
    public void deleteUriRights(ActualUri uri) {
        try {
            collection.remove(new BasicDBObject(COLUMN_URI_HASH, MongoDbUtils.calculateMd5IdKeyByActualUri(uri)));
            LoggerEventsRecorder.saveCleanupUriEvent(COLLECTION, uri);
        } catch (Exception exc) {
            logger.error("Unable to delete URL rights for '" + uri + "': " + exc, exc);
            LoggerEventsRecorder.saveCleanupUriFailedEvent(COLLECTION, exc, uri);
        }
    }

    private Option<Instant> getOldestTimestamp() {
        return collection
                .find(new BasicDBObject(), SqlOrder.orderByColumn(COLUMN_TIMESTAMP), SqlLimits.first(1))
                .firstO()
                .map(UriRights::getTimestamp);
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp) {
        logger.info("deleteByTimestampLessBatch({})", timestamp);
        getOldestTimestamp().ifPresent(t -> logger.info("oldest record before cleanup: {}", t));
        DBObject query = new BasicDBObject(COLUMN_TIMESTAMP, new BasicDBObject("$lt", timestamp.toDate()));
        BulkWriteOperation builder = collection.getCollection().initializeUnorderedBulkOperation();
        builder.find(query).remove();
        builder.execute();
        getOldestTimestamp().ifPresent(t -> logger.info("oldest record after cleanup: {}", t));
    }

    @Override
    public boolean findExistsUriRight(ActualUri uri, PassportUidOrZero uid) {
        return collection.exists(getIdKeyObject(uri, uid));
    }

    @Override
    public Option<ActualUri> findUriByFileIdAndUid(String fileId, PassportUidOrZero uid) {
        BasicDBObject query = new BasicDBObject(Cf.<String, Object>map(
                COLUMN_UID, uid.getUid(), COLUMN_FILE_ID, fileId));
        return collection.find(query, new BasicDBObject(COLUMN_URI, 1),
                SqlOrder.orderByColumnDesc(COLUMN_TIMESTAMP), SqlLimits.first(1),
                MongoDbUtils.<String>getValueF(COLUMN_URI).andThen(ActualUri.consFromStringF())).firstO();
    }

    @Override
    public boolean validate(String fileId, PassportUidOrZero uid) {
        val query = new BasicDBObject(Cf.<String, Object>map(COLUMN_UID, uid.getUid(), COLUMN_FILE_ID, fileId));
        return collection.exists(query) || collection.findCount(query, ReadPreference.primary()) > 0;
    }

    @Override
    public ListF<ActualUri> findUrisAccessedByUid(PassportUidOrZero uid) {
        return collection.find(new BasicDBObject(COLUMN_UID, uid.getUid()), new BasicDBObject(COLUMN_URI, 1),
                MongoDbUtils.<String>getValueF(COLUMN_URI))
                .map(ActualUri::new);
    }

    @Override
    public void saveOrUpdateUriRight(ActualUri uri, PassportUidOrZero uid) {
        BasicDBObject query = getIdKeyObject(uri, uid);

        BasicDBObject toSet = new BasicDBObject();
        toSet.put(COLUMN_URI, uri.getUriString());
        toSet.put(COLUMN_URI_HASH, MongoDbUtils.calculateMd5IdKeyByActualUri(uri));
        toSet.put(COLUMN_UID, uid.getUid());
        toSet.put(COLUMN_TIMESTAMP, new Date());
        BasicDBObject update = new BasicDBObject("$set", toSet);

        collection.update(query, update, true, false);

        logger.info("Access to URL '{}' for '{}' stored", uri, uid);
    }

    @Override
    public void updateUriRights(ActualUri uri, String fileId) {
        DBObject clause1 = new BasicDBObject(COLUMN_URI_HASH, MongoDbUtils.calculateMd5IdKeyByActualUri(uri));
        DBObject clause2 = new BasicDBObject(COLUMN_FILE_ID, new BasicDBObject("$ne", fileId));
        BasicDBList and = new BasicDBList();
        and.add(clause1);
        and.add(clause2);
        BasicDBObject query = new BasicDBObject("$and", and);

        BasicDBObject toSet = new BasicDBObject();
        toSet.put(COLUMN_FILE_ID, fileId);
        BasicDBObject update = new BasicDBObject("$set", toSet);

        collection.update(query, update, false, true);
    }
}
