package ru.yandex.chemodan.app.docviewer.dao.schedule;

import java.util.Date;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;

/**
 * @author akirakozov
 */
public class MongoScheduledTaskDao implements ScheduledTaskDao {
    private static final String COLLECTION_PREFIX = "schedule";

    private static final String COLUMN_ID = "_id";

    private static final String COLUMN_HOSTNAME = "hostname";
    private static final String COLUMN_TIMESTAMP = "timestamp";

    private final String name;

    private final MongoDbAdapter mongoDbAdapter;


    public MongoScheduledTaskDao(String name, MongoDbAdapter mongoDbAdapter) {
        this.mongoDbAdapter = mongoDbAdapter;
        this.name = name;
    }

    private DBCollection getCollection() {
        return mongoDbAdapter.getDatabase().getCollection(COLLECTION_PREFIX + "-" + name);
    }

    private DBObject getIdKeyObject(String taskId) {
        return new BasicDBObject(COLUMN_ID, MongoDbUtils.calculateMd5IdKey(taskId));
    }

    private ScheduledTask deserialize(DBObject dbObject) {

        ScheduledTask scheduleItem = new ScheduledTask();

        scheduleItem.setHostname(MongoDbUtils.getString(dbObject, COLUMN_HOSTNAME, true, false));
        scheduleItem.setTimestamp(MongoDbUtils.getInstant(dbObject, COLUMN_TIMESTAMP).getOrThrow(COLUMN_TIMESTAMP));

        return scheduleItem;
    }

    private Function<DBObject, ScheduledTask> deserializeF() {
        return this::deserialize;
    }

    @Override
    public Option<ScheduledTask> find(String taskId) {
        return MongoDbUtils.findOne(getCollection(), getIdKeyObject(taskId), deserializeF());
    }

    @Override
    public void saveOrUpdateScheduleItem(String taskId, String hostname) {
        DBObject query = getIdKeyObject(taskId);

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_HOSTNAME, hostname);
        toUpdate.put(COLUMN_TIMESTAMP, new Date());
        final DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, true, false);
    }

    @Override
    public void delete(String taskId) {
        getCollection().remove(getIdKeyObject(taskId));
    }

}
