package ru.yandex.chemodan.app.docviewer.dao.sessions;

import java.util.Date;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;

/**
 * @author ssytnik
 */
public class MongoSessionDao implements SessionDao {

    private static final int SESSION_ID_LENGTH = 16;

    public final DynamicProperty<Integer> sessionTtl =
            new DynamicProperty<>("docviewer.session.record.cleanup.ttl.minutes", 10);

    public static final String COLLECTION = "sessions";

    private static final String COLUMN_ID = "_id";
    private static final String SUBCOLUMN_SESSION = "session";
    private static final String SUBCOLUMN_KEY = "key";

    private static final String COLUMN_VALUE = "value";
    private static final String COLUMN_TIMESTAMP = "timestamp";

    @Autowired
    @Qualifier("mongoDbAdapter")
    private MongoDbAdapter mongoDbAdapter;

    private DBCollection getCollection() {
        return mongoDbAdapter.getDatabase().getCollection(COLLECTION);
    }

    private DBObject getIdKeyObject(String sessionId, SessionKey key) {
        DBObject id = new BasicDBObject();
        id.put(SUBCOLUMN_SESSION, sessionId);
        id.put(SUBCOLUMN_KEY, key.getPrefixAndKey());
        return new BasicDBObject(COLUMN_ID, id);
    }

    private DBObject getTimestampQuery(boolean gtOp, Instant timestamp) {
        String op = gtOp ? "$gt" : "$lt";
        return new BasicDBObject(op, timestamp.toDate());
    }

    private DBObject getValidRecordQuery(String sessionId, SessionKey key) {
        DBObject res = getIdKeyObject(sessionId, key);
        res.put(COLUMN_TIMESTAMP, getTimestampQuery(true, new Instant().minus(getSessionTtl())));
        return res;
    }


    @Override
    public Option<String> findValidValue(String sessionId, SessionKey key) {
        if (StringUtils.isEmpty(sessionId)) {
            return Option.empty();
        }

        return MongoDbUtils.findOne(getCollection(), getValidRecordQuery(sessionId, key),
                MongoDbUtils.<String>getValueF(COLUMN_VALUE).andThen(key.toRawValueF()));
    }

    @Override
    public String createOrUpdate(String sessionId, SessionKey key, String value) {
        if (StringUtils.isEmpty(sessionId)) {
            sessionId = Random2.R.nextAlnum(SESSION_ID_LENGTH);
        }

        DBObject query = getIdKeyObject(sessionId, key);

        DBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_VALUE, key.toStoredValue(value));
        toUpdate.put(COLUMN_TIMESTAMP, new Date());
        final DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, true, false);

        return sessionId;
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp) {
        DBObject query = new BasicDBObject(COLUMN_TIMESTAMP,  getTimestampQuery(false, timestamp));

        getCollection().remove(query);
    }


    // for tests

    @Override
    public long count() {
        return getCollection().count();
    }

    @Override
    public long count(String sessionId, SessionKey key) {
        return getCollection().count(getIdKeyObject(sessionId, key));
    }

    public Duration getSessionTtl() {
        return Duration.standardMinutes(sessionTtl.get());
    }
}
