package ru.yandex.chemodan.app.docviewer.dao.sessions;

import lombok.AllArgsConstructor;
import lombok.Data;

import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.crypt.AesCrypt;

/**
 * Immutable session record key which
 * also encapsulates its value format
 *
 * @author ssytnik
 */
@Data
@AllArgsConstructor
public abstract class SessionKey {

    private final String prefix;
    private final String key;

    public String getPrefixAndKey() {
        return prefix + ":" + key;
    }

    public String getKey() {
        return key;
    }

    public String toStoredValue(String rawValue) {
        return rawValue;
    }

    public String toRawValue(String storedValue) {
        return storedValue;
    }

    public static String toHashValue(String rawValue) {
        return MongoDbUtils.calculateMd5IdKey(rawValue);
    }

    public Function<String, String> toRawValueF() {
        return this::toRawValue;
    }


    public static class SessionPlain extends SessionKey {
        public static final String PREFIX = "plain";

        public SessionPlain(String key) {
            super(PREFIX, key);
        }

    }


    private static abstract class SessionPassword extends SessionKey {

        public SessionPassword(String prefix, String key) {
            super(prefix, key);
        }

        @Override
        public String toStoredValue(String rawValue) {
            return AesCrypt.encrypt(rawValue);
        }

        @Override
        public String toRawValue(String storedValue) {
            return AesCrypt.decrypt(storedValue);
        }
    }

    public static class SessionCopyPassword extends SessionPassword {
        public static final String PREFIX = "copy-password";

        public SessionCopyPassword(String key) {
            super(PREFIX, key);
        }

        public SessionCopyPassword(ActualUri uri) {
            this(uri.getUriString());
        }

    }

    public static class SessionConvertPassword extends SessionPassword {
        public static final String PREFIX = "convert-password";

        public SessionConvertPassword(String fileId) {
            super(PREFIX, fileId);
        }

    }

}
