package ru.yandex.chemodan.app.docviewer.dao.uris;

import java.util.Date;
import java.util.Map;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.BulkWriteOperation;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class MongoStoredUriDao implements StoredUriDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoStoredUriDao.class);

    public static final String COLLECTION = "uris";

    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_CONTENT_TYPE = "content-type";
    private static final String COLUMN_CONVERT_TARGETS = "convert-targets";
    @Deprecated
    private static final String COLUMN_FILE_ID = "file-id";
    private static final String COLUMN_TIMESTAMP = "timestamp";

    private static final String COLUMN_URI = "uri";

    private static final String COLUMN_ERROR = "error";
    private static final String COLUMN_ERROR_CODE = "error-code";
    private static final String COLUMN_ERROR_ARCHIVE_PATH = "error-archive-path";

    private static final String COLUMN_PASSWORDS = "passwords";
    private static final String COLUMN_EXTENDED_INFO = "ext";
    private static final String COLUMN_CONTENT_SIZE = "size";
    private static final String COLUMN_MAX_CONTENT_SIZE = "max-size";

    private static final String SUBCOLUMN_PATH = "path";
    private static final String SUBCOLUMN_HASH = "hash";

    private static final Integer MARKIT = 1;

    @Autowired
    @Qualifier("mongoDbAdapter")
    private MongoDbAdapter mongoDbAdapter;

    @Value("${cleanup.threads}")
    private int cleanupThreads;


    public DBCollection getCollection() {
        return mongoDbAdapter.getDatabase().getCollection(COLLECTION);
    }

    @Override
    public void delete(ActualUri uri) {
        try {
            DBObject query = getIdKeyObject(uri);
            getCollection().remove(query);
            LoggerEventsRecorder.saveCleanupUriEvent(COLLECTION, uri);
        } catch (Exception exc) {
            LoggerEventsRecorder.saveCleanupUriFailedEvent(COLLECTION, exc, uri);
            logger.error("Unable to delete URI info '" + uri + "': " + exc, exc);
        }
    }

    private StoredUri deserialize(DBObject dbObject) {
        StoredUri result = new StoredUri();
        result.setUri(new ActualUri((String) dbObject.get(COLUMN_URI)));

        if (dbObject.containsField(COLUMN_CONTENT_TYPE)) {
            result.setContentType((String) dbObject.get(COLUMN_CONTENT_TYPE));
        }
        result.setFileId(MongoDbUtils.getString(dbObject, COLUMN_FILE_ID, true, false));
        if (dbObject.containsField(COLUMN_CONVERT_TARGETS)) {
            result.setConvertTargets(Cf.x(MongoDbUtils.getMap(dbObject, COLUMN_CONVERT_TARGETS,
                    obj -> MongoDbUtils.getEnum(obj, TargetType.getResolver()),
                    MongoDbUtils::getFloat)));
        }
        result.setError(MongoDbUtils.getString(dbObject, COLUMN_ERROR, true, false));
        if (dbObject.containsField(COLUMN_ERROR_CODE)) {
            result.setErrorCode(Option.of(ErrorCode.valueOf(
                    String.valueOf(dbObject.get(COLUMN_ERROR_CODE)), ErrorCode.UNKNOWN_COPY_ERROR)));
        }
        if (dbObject.containsField(COLUMN_ERROR_ARCHIVE_PATH)) {
            result.setErrorArchivePath(MongoDbUtils.getString(dbObject, COLUMN_ERROR_ARCHIVE_PATH, false, true));
        }
        if (dbObject.containsField(COLUMN_TIMESTAMP)) {
            result.setTimestamp(MongoDbUtils.getInstant(dbObject, COLUMN_TIMESTAMP).getOrThrow(COLUMN_TIMESTAMP));
        }
        if (dbObject.containsField(COLUMN_PASSWORDS)) {
            result.setPasswords(deserializePasswords(dbObject.get(COLUMN_PASSWORDS)));
        }
        if (dbObject.containsField(COLUMN_CONTENT_SIZE)) {
            Option<Long> size = MongoDbUtils.getLong(dbObject, COLUMN_CONTENT_SIZE);
            if (size.isPresent()) {
                result.setContentSize(DataSize.fromBytes(size.get()));
            }
        }
        if (dbObject.containsField(COLUMN_MAX_CONTENT_SIZE)) {
            Option<Long> size = MongoDbUtils.getLong(dbObject, COLUMN_MAX_CONTENT_SIZE);
            if (size.isPresent()) {
                result.setMaxContentSize(DataSize.fromBytes(size.get()));
            }
        }
        if (dbObject.containsField(COLUMN_EXTENDED_INFO)) {
            result.setExtInfo(Cf.hashMap(MongoDbUtils.getMap(dbObject, COLUMN_EXTENDED_INFO)));
        }
        return result;
    }

    private Tuple2List<String, String> deserializePasswords(Object passwords) {
        Validate.isTrue(passwords instanceof BasicDBList, "DBObject " + passwords + "' has incorrect structure");

        Tuple2List<String, String> res = Tuple2List.arrayList();
        for (Object entryObj : (BasicDBList) passwords) {
            DBObject entry = (DBObject) entryObj;
            res.add(MongoDbUtils.getStringWithEmpties(entry, SUBCOLUMN_PATH).get(),
                    MongoDbUtils.getStringNoEmpties(entry, SUBCOLUMN_HASH).get());
        }
        return sortPasswords(res);
    }

    private Tuple2List<String, String> sortPasswords(Tuple2List<String, String> passwords) {
        return passwords.sortedBy(Tuple2.<String, String>get1F().andThen(Cf.String.lengthF()));
    }

    private BasicDBList serializePasswords(Tuple2List<String, String> passwords) {
        BasicDBList res = new BasicDBList();
        for (Tuple2<String, String> entry : sortPasswords(passwords)) {
            res.add(new BasicDBObject(Cf.map(SUBCOLUMN_PATH, entry.get1(), SUBCOLUMN_HASH, entry.get2())));
        }
        return res;
    }

    @Override
    public Option<StoredUri> find(ActualUri uri) {
        return MongoDbUtils.findOne(getCollection(), getIdKeyObject(uri), this::deserialize);
    }

    @Override
    public ListF<StoredUri> findAllByFileId(String fileId) {
        DBObject query = new BasicDBObject(COLUMN_FILE_ID, fileId);
        return MongoDbUtils.find(getCollection(), query, this::deserialize);
    }

    @Override
    public Tuple2List<String, String> getPasswordsSorted(ActualUri uri) {
        return find(uri).get().getPasswords();
    }

    @Override
    public Option<Tuple2List<String, String>> getPasswordsSortedO(ActualUri uri) {
        return find(uri).map(StoredUri::getPasswords);
    }

    @Override
    public void updatePasswords(ActualUri uri, Tuple2List<String, String> passwords) {
        logger.debug("Setting passwords of URI '{}'", uri);

        DBObject query = getIdKeyObject(uri);

        final BasicDBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_PASSWORDS, serializePasswords(passwords));
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void updatePasswordRaw(ActualUri uri, String archivePath, String rawValue) {
        Option<Tuple2List<String, String>> passwordsO = getPasswordsSortedO(uri);
        if (passwordsO.isEmpty()) {
            logger.debug("URI '{}' not found, don't update password for path '{}'", uri, archivePath);
            return;
        }

        Tuple2List<String, String> passwords = passwordsO.get();
        String newHashValue = SessionKey.toHashValue(rawValue);

        Tuple2<Tuple2List<String, String>, Tuple2List<String, String>> partition =
                passwords.partitionBy1(Cf.String.equalsF(archivePath));

        if (partition.get1().isNotEmpty()) {
            if (partition.get1().single().get2().equals(newHashValue)) {
                return;
            }
            passwords = partition.get2();
        }
        passwords = passwords.plus1(archivePath, newHashValue);

        updatePasswords(uri, passwords);

        logger.debug("Password updated for URI '{}' and path '{}'", uri, archivePath);
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp) {
        DBObject query = new BasicDBObject(COLUMN_TIMESTAMP, new BasicDBObject("$lt", timestamp.toDate()));
        BulkWriteOperation builder = getCollection().initializeUnorderedBulkOperation();
        builder.find(query).remove();
        builder.execute();
    }

    @Override
    public void deleteErrorsByTimestampLessBatch(Instant timestamp) {

        BasicDBList or = new BasicDBList();
        or.add(new BasicDBObject(COLUMN_ERROR, new BasicDBObject("$exists", true)));
        or.add(new BasicDBObject(COLUMN_ERROR_CODE, new BasicDBObject("$exists", true)));
        or.add(new BasicDBObject(COLUMN_ERROR_ARCHIVE_PATH, new BasicDBObject("$exists", true)));

        BasicDBList and = new BasicDBList();
        and.add(new BasicDBObject(COLUMN_TIMESTAMP, new BasicDBObject("$lt", timestamp.toDate())));
        and.add(new BasicDBObject("$or", or));


        BulkWriteOperation builder = getCollection().initializeUnorderedBulkOperation();
        builder.find(new BasicDBObject("$and", and)).remove();
        builder.execute();
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp, final Function1V<StoredUri> deleteHandler) {
        DBObject query = new BasicDBObject(COLUMN_TIMESTAMP, new BasicDBObject("$lt", timestamp.toDate()));
        MongoDbUtils.forEachBatch(getCollection(), query, COLUMN_TIMESTAMP, cleanupThreads,
                a -> {
                    try {
                        StoredUri storedUri = null;
                        try {
                            storedUri = deserialize(a);
                        } catch (Exception e) {
                            logger.debug("Couldn't deserialize stored uri: " + e.getMessage());
                            MongoDbUtils.removeRecordById(getCollection(), (String) a.get(COLUMN_ID));
                        }

                        if (storedUri != null) {
                            deleteHandler.apply(storedUri);
                        }
                    } catch (Exception e) {
                        logger.debug("Couldn't remove record: " + a, e);
                    }
                });
    }

    private DBObject getIdKeyObject(ActualUri uri) {
        DBObject query = new BasicDBObject();
        query.put(COLUMN_ID, MongoDbUtils.calculateMd5IdKeyByActualUri(uri));
        return query;
    }

    @Override
    public void saveOrUpdateUri(ActualUri uri, Option<String> contentType,
            TargetType convertTargetType, Float priority)
    {
        logger.debug("Creating URI '{}' with target '{}' and priority '{}'", uri, convertTargetType, priority);

        DBObject query = getIdKeyObject(uri);

        {
            DBObject toUpdate = new BasicDBObject();
            toUpdate.put(COLUMN_URI, uri.getUriString());
            toUpdate.put(COLUMN_TIMESTAMP, new Date());

            if (contentType.isPresent())
                toUpdate.put(COLUMN_CONTENT_TYPE, contentType.get());

            DBObject update = new BasicDBObject("$set", toUpdate);
            getCollection().update(query, update, true, false);
        }

        {
            // XXX: need to increase only
            DBObject toSet = new BasicDBObject();
            toSet.put(COLUMN_CONVERT_TARGETS + "." + convertTargetType.name(), priority);
            DBObject set = new BasicDBObject("$set", toSet);
            getCollection().update(query, set, true, false);
        }

        {
            DBObject toUpdate = new BasicDBObject();
            toUpdate.put(COLUMN_TIMESTAMP, new Date());
            DBObject update = new BasicDBObject("$set", toUpdate);
            getCollection().update(query, update, false, false);
        }
    }

    @Override
    public void updateUri(ActualUri uri, ErrorCode errorCode, String error, Option<String> errorArchivePath,
            Option<Long> contentSize, Option<Long> maxFileSize)
    {
        logger.debug("Setting error code and text of URI '{}' to '{}' and '{}'",
                uri, errorCode, error);

        DBObject query = getIdKeyObject(uri);

        final BasicDBObject toUpdate = new BasicDBObject();
        toUpdate.put(COLUMN_ERROR, error);
        toUpdate.put(COLUMN_ERROR_CODE, errorCode.name());
        if (errorArchivePath.isPresent()) {
            toUpdate.put(COLUMN_ERROR_ARCHIVE_PATH, errorArchivePath.get());
        }
        if (contentSize.isPresent()) {
            toUpdate.put(COLUMN_CONTENT_SIZE, contentSize.get());
        }
        if (maxFileSize.isPresent()) {
            toUpdate.put(COLUMN_MAX_CONTENT_SIZE, maxFileSize.get());
        }
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void updateUri(ActualUri uri, Option<String> reportedContentType,
        String fileId, DataSize size, Option<Instant> serpLastAccess)
    {
        MapF<String, Object> extInfo =
                serpLastAccess.map(sla -> Cf.map(StoredUri.SERP_LAST_ACCESS, (Object) sla)).getOrElse(Cf.map());

        logger.debug("Setting content type and file ID of URI '{}' to '{}' and '{}'",
                uri, reportedContentType, fileId);

        DBObject query = getIdKeyObject(uri);

        DBObject toUpdate = new BasicDBObject();
        if (reportedContentType.isPresent()) {
            toUpdate.put(COLUMN_CONTENT_TYPE, reportedContentType.get());
        }
        toUpdate.put(COLUMN_FILE_ID, fileId);
        toUpdate.put(COLUMN_CONTENT_SIZE, size.toBytes());
        if (!extInfo.isEmpty()) {
            toUpdate.put(COLUMN_EXTENDED_INFO, serializeExtInfo(extInfo));
        }
        DBObject update = new BasicDBObject("$set", toUpdate);

        getCollection().update(query, update, false, false);
    }

    @Override
    public void updateUriClean(ActualUri uri) {
        logger.debug("Clearing copy result information about '{}'", uri);

        DBObject query = getIdKeyObject(uri);

        final BasicDBObject toUnset = new BasicDBObject();
        toUnset.put(COLUMN_CONTENT_TYPE, MARKIT);
        toUnset.put(COLUMN_FILE_ID, MARKIT);
        toUnset.put(COLUMN_ERROR, MARKIT);
        toUnset.put(COLUMN_ERROR_CODE, MARKIT);
        toUnset.put(COLUMN_ERROR_ARCHIVE_PATH, MARKIT);

        DBObject update = new BasicDBObject("$unset", toUnset);

        getCollection().update(query, update, false, false);
    }

    private BasicDBObject serializeExtInfo(MapF<String, Object> extInfo) {
        BasicDBObject infoObj = new BasicDBObject();
        for (Map.Entry<String, Object> e : extInfo.entrySet()) {
            Object val;
            if (e.getValue() instanceof String) {
                val = e.getValue();
            } else if (e.getValue() instanceof Instant) {
                val = new Date(((Instant) e.getValue()).getMillis());
            } else {
                throw new IllegalArgumentException("Unsupported value: " + String.valueOf(e.getValue()));
            }
            infoObj.append(e.getKey(), val);
        }
        return infoObj;
    }

}
