package ru.yandex.chemodan.app.docviewer.dao.user;

import java.util.List;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBObject;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.rights.MongoUriRightsDao;
import ru.yandex.commune.mongo.MongoBenderParserSerializer;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.commune.mongo.MongoHolder;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.time.TimeUtils;

public class MongoUserDocumentsDao implements UserDocumentsDao {

    static final String COLLECTION = "user-documents";
    static final String COLUMN_ID = "_id";
    static final String COLUMN_UID = "uid";
    static final String COLUMN_URI = "uri";
    static final String COLUMN_PREVIEW_URI = "preview-file-link";
    static final String COLUMN_LAST_ACCESS_TIME = "last-access-time";
    static final String TYPE = "type";

    @Value("${cleanup.threads}")
    private int cleanupThreads;

    private final MongoCollection<String, StoredUserDocument> collection;
    private final MongoBenderParserSerializer<StoredUserDocument> bender;

    public MongoUserDocumentsDao(DB db) {
        bender = MongoBenderParserSerializer.cons(StoredUserDocument.class, MongoHolder.defaultBenderConfiguration());
        collection = new MongoCollection<>(
                db.getCollection(COLLECTION), StoredUserDocument.class,
                MongoHolder.defaultBenderConfiguration(),
                MongoDefaultInterceptors.defaultInterceptors());
    }

    @Override
    public ListF<StoredUserDocument> findDocumentByUrl(ActualUri uri) {
        return collection.find(
                new BasicDBObject(Cf.map(COLUMN_URI, uri.getUriString())),
                SqlOrder.unordered(),
                SqlLimits.first(1));
    }

    @Override
    public void deleteByLastAccessLessBatch(Instant timestamp, Function1V<StoredUserDocument> deleteHandler) {
        DBObject query = new BasicDBObject(COLUMN_LAST_ACCESS_TIME, new BasicDBObject("$lt", timestamp.toDate()));
        MongoDbUtils.forEachBatch(collection.getCollection(), query, COLUMN_LAST_ACCESS_TIME,
                cleanupThreads, bender.parseF().andThen(deleteHandler));
    }

    @Override
    public List<StoredUserDocument> findDocuments(PassportUidOrZero uid, int limit, int offset) {
        return collection.find(
                new BasicDBObject(Cf.map(COLUMN_UID, uid.getUid())),
                SqlOrder.orderByColumnDesc(COLUMN_LAST_ACCESS_TIME),
                SqlLimits.range(offset, limit));
    }

    @Override
    public Option<StoredUserDocument> findDocument(PassportUidOrZero uid, ActualUri uri) {
        return collection.findByRawId(MongoUriRightsDao.generateId(uri, uid)).firstO();
    }

    @Override
    public void saveOrUpdateDocument(StoredUserDocument document) {
        collection.insertOrUpdate(document);
    }

    @Override
    public void updateAccessTime(PassportUidOrZero uid, ActualUri uri) {

        collection.update(
                MongoUriRightsDao.getIdKeyObject(uri, uid),
                new BasicDBObject(Cf.map("$set", Cf.map(COLUMN_LAST_ACCESS_TIME, MongoUtils.toMongoValue(TimeUtils.now())))));
    }

    @Override
    public void delete(StoredUserDocument storedUserDocument) {
        collection.remove(storedUserDocument.getId());
    }

    @Override
    public void delete(PassportUidOrZero uid, ActualUri uri) {
        collection.remove(MongoUriRightsDao.getIdKeyObject(uri, uid));
    }

}
