package ru.yandex.chemodan.app.docviewer.dao.ydb;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.auth.tvm.TvmAuthContext;
import com.yandex.ydb.auth.tvm.YdbClientId;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.docviewer.dao.cleanup.CleanerTaskDao;
import ru.yandex.chemodan.app.docviewer.dao.cleanup.MongoCleanerTaskDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.MongoImageDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.MongoPdfWarmupDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupDao;
import ru.yandex.chemodan.app.docviewer.dao.results.MongoStoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.rights.MongoUriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.schedule.MongoScheduledTaskDao;
import ru.yandex.chemodan.app.docviewer.dao.schedule.ScheduledTaskDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.MongoStoredUriDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.ydb.dao.OneTableYdbDao;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbTimeoutSettings;
import ru.yandex.chemodan.ydb.dao.YdbUtils;
import ru.yandex.chemodan.ydb.dao.pojo.YdbTestUtils;
import ru.yandex.chemodan.ydb.dao.twin.TwinDaoConfiguration;
import ru.yandex.chemodan.ydb.dao.twin.TwinDaoFactory;
import ru.yandex.chemodan.ydb.dao.twin.TwinDaoInvocationHandler;
import ru.yandex.chemodan.ydb.dao.twin.TwinDaoMode;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.devtools.test.YaTest;
import ru.yandex.misc.env.EnvironmentType;

/**
 * @author yashunsky
 */
@Configuration
public class YdbContextConfiguration {
    private final DynamicProperty<Long> ydbQueryTimeoutMillis =
            new DynamicProperty<>("docviewer.ydb.query.timeout.seconds", 3000L);
    private final DynamicProperty<Long> ydbQueryOperationTimeoutDeltaMillis =
            new DynamicProperty<>("docviewer.ydb.query.operation.timeout.delta.millis", 100L);
    private final DynamicProperty<Long> ydbTxTimeoutMillis =
            new DynamicProperty<>("docviewer.ydb.tx.timeout.seconds", 3000L);
    private final DynamicProperty<Long> ydbTxOperationTimeoutDeltaMillis =
            new DynamicProperty<>("docviewer.ydb.tx.operation.timeout.delta.millis", 100L);

    @Bean
    public YdbTimeoutSettings timeoutSettings() {
        return new YdbTimeoutSettings(ydbQueryTimeoutMillis::get, ydbQueryOperationTimeoutDeltaMillis::get,
                ydbTxTimeoutMillis::get, ydbTxOperationTimeoutDeltaMillis::get);
    }

    @Bean
    public ThreadLocalYdbTransactionManager ydbTransactionManager(
            TableClient tableClient,
            YdbTimeoutSettings timeoutSettings,
            @Value("${ydb.retry-count}") int retryCount,
            @Value("${ydb.session-retry-context-threads}") int sessionRetryContextThreads)
    {
        return new ThreadLocalYdbTransactionManager(tableClient, timeoutSettings, retryCount, sessionRetryContextThreads);
    }

    @Bean(destroyMethod = "close")
    public TvmAuthContext ydbTvmAuthContext(
        @Value("${tvm.self.client-id}") int selfClientId,
        @Value("${tvm.self.client-secret}") String selfClientSecret)
    {
        return TvmAuthContext.useTvmApi(selfClientId, selfClientSecret);
    }

    @Bean(destroyMethod = "close")
    public TableClient tableClient(
            TvmAuthContext ydbTvmAuthContext,
            @Value("${ydb.endpoint}") String endpoint,
            @Value("${ydb.db}") String database,
            @Value("${ydb.pool-size.min}") int poolSizeMin,
            @Value("${ydb.pool-size.max}") int poolSizeMax)
    {
        RpcTransport transport;

        if (YaTest.insideYaTest) {
            endpoint = System.getenv("YDB_ENDPOINT");
            database = System.getenv("YDB_DATABASE");
            transport = GrpcTransport.forEndpoint(endpoint, database).build();
        } else {
            transport = GrpcTransport.forEndpoint(endpoint, database)
                    .withAuthProvider(ydbTvmAuthContext.authProvider(YdbClientId.YDB))
                    .build();
        }

        TableClient tableClient = TableClient.newClient(GrpcTableRpc.ownTransport(transport))
                .sessionPoolSize(poolSizeMin, poolSizeMax)
                .build();

        if (YaTest.insideYaTest) {
            String finalDatabase = database;
            YdbTimeoutSettings timeoutSettings = YdbTestUtils.getTestTimeoutSettings();
            ThreadLocalYdbTransactionManager transactionManager = new ThreadLocalYdbTransactionManager(tableClient, timeoutSettings);
            ListF<OneTableYdbDao> daos = Cf.list(
                    new YdbImageDao(transactionManager),
                    new YdbStoredResultDao(transactionManager, null, Duration.standardHours(1)),
                    new YdbStoredUriDao(transactionManager, Duration.standardHours(1)),
                    new YdbUriRightsDao(transactionManager, Duration.standardHours(1)));
            transactionManager.executeInTmpSession((session, txControl) -> {
                YdbUtils.createTablesIfMissing(finalDatabase, session, daos);
                return CompletableFuture.completedFuture(Result.success(null));
            });
        }

        return tableClient;
    }

    @Bean
    public YdbImageDao ydbImageDao(ThreadLocalYdbTransactionManager ydbTransactionManager) {
        return new YdbImageDao(ydbTransactionManager);
    }

    @Bean
    public YdbStoredResultDao ydbStoredResultDao(
            ThreadLocalYdbTransactionManager ydbTransactionManager, AliveAppInfo aliveAppInfo,
            @Value("${ydb.stored.result.ttl}") Duration ttl) {
        return new YdbStoredResultDao(ydbTransactionManager, aliveAppInfo, ttl);
    }

    @Bean
    public YdbStoredUriDao ydbStoredUriDao(ThreadLocalYdbTransactionManager ydbTransactionManager,
                                           @Value("${ydb.stored.uri.ttl}") Duration ttl)
    {
        return new YdbStoredUriDao(ydbTransactionManager, ttl);
    }

    @Bean
    public YdbUriRightsDao ydbUriRightsDao(ThreadLocalYdbTransactionManager ydbTransactionManager,
                                           @Value("${ydb.uri.right.ttl}") Duration ttl) {
        return new YdbUriRightsDao(ydbTransactionManager, ttl);
    }

    @Bean
    public YdbCleanerTaskDao ydbCleanerTaskDao(ThreadLocalYdbTransactionManager ydbTransactionManager) {
        return new YdbCleanerTaskDao(ydbTransactionManager);
    }

    @Bean
    public YdbPdfWarmupDao ydbPdfWarmupDao(
            ThreadLocalYdbTransactionManager ydbTransactionManager,
            TwinDaoConfiguration twinDaoConfiguration,
            @Value("${pdf.image.warmup.blockSize}") int blockSize,
            @Value("${pdf.image.warmup.ttl}") Duration ttl)
    {
        return new YdbPdfWarmupDao(ydbTransactionManager,
                blockSize, ttl, () -> twinDaoConfiguration.getMode() == TwinDaoMode.SECONDARY);
    }

    @Bean
    public YdbScheduledTaskDao ydbCopierScheduledTaskDao(
            ThreadLocalYdbTransactionManager ydbTransactionManager,
            @Value("copier") String copierName)
    {
        return new YdbScheduledTaskDao(ydbTransactionManager, copierName);
    }

    @Bean
    public YdbScheduledTaskDao ydbConvertScheduledTaskDao(
            ThreadLocalYdbTransactionManager ydbTransactionManager,
            @Value("convert") String convertName)
    {
        return new YdbScheduledTaskDao(ydbTransactionManager, convertName);
    }

    @Bean
    public TwinDaoConfiguration twinDaoConfiguration(
            @Value("${ydb.twin-dao.mode}") TwinDaoMode mode,
            @Value("${ydb.twin-dao.threads-count}") int threadsCount,
            @Value("${ydb.twin-dao.queue-size}") int queueSize)
    {
        return new TwinDaoConfiguration(mode, threadsCount, queueSize, "ydb" + getTwinDaoModeSuffix());
    }

    private String getTwinDaoModeSuffix() {
        return EnvironmentType.getActiveSecondary().map(env -> "-" + env.toLowerCase()).getOrElse("");
    }

    @Bean
    public ImageDao pdfImageDao(
            YdbImageDao ydbDao,
            MongoImageDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(ImageDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                TwinDaoInvocationHandler.EMPTY_RESULT);
    }

    @Bean
    public StoredResultDao storedResultDao(
            YdbStoredResultDao ydbDao,
            MongoStoredResultDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(StoredResultDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                TwinDaoInvocationHandler.EMPTY_RESULT);
    }

    @Bean
    public StoredUriDao storedUriDao(
            YdbStoredUriDao ydbDao,
            MongoStoredUriDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(StoredUriDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                TwinDaoInvocationHandler.EMPTY_RESULT);
    }

    @Bean
    public UriRightsDao uriRightsDao(
            YdbUriRightsDao ydbDao,
            MongoUriRightsDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        SetF<String> returningBooleanMethodNames = Cf.set("findExistsUriRight", "validate");
        return TwinDaoFactory.createProxyDao(UriRightsDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                (methodName, result) ->
                        TwinDaoInvocationHandler.EMPTY_RESULT.apply(methodName, result)
                                || (Boolean.FALSE.equals(result) && returningBooleanMethodNames.containsTs(methodName)));
    }

    @Bean
    public CleanerTaskDao cleanerTaskDao(
            YdbCleanerTaskDao ydbDao,
            MongoCleanerTaskDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(CleanerTaskDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                (methodName, result) ->
                        TwinDaoInvocationHandler.EMPTY_RESULT.apply(methodName, result)
                                || (Boolean.TRUE.equals(result) && "saveAttempt".equals(methodName)));
    }

    @Bean
    public PdfWarmupDao pdfWarmupDao(
            YdbPdfWarmupDao ydbDao,
            MongoPdfWarmupDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(PdfWarmupDao.class, mongoDao, ydbDao, twinDaoConfiguration,
                (methodName, result) -> ("createTasks".equals(methodName)
                        && result instanceof Collection
                        && !((Collection) result).isEmpty()));
    }

    @Bean
    public ScheduledTaskDao copierScheduledTaskDao(
            @Qualifier("ydbCopierScheduledTaskDao") YdbScheduledTaskDao ydbDao,
            @Qualifier("mongoCopierScheduledTaskDao") MongoScheduledTaskDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(
                ScheduledTaskDao.class, mongoDao, ydbDao, twinDaoConfiguration, TwinDaoInvocationHandler.EMPTY_RESULT);
    }

    @Bean
    public ScheduledTaskDao convertScheduledTaskDao(
            @Qualifier("ydbConvertScheduledTaskDao") YdbScheduledTaskDao ydbDao,
            @Qualifier("mongoConvertScheduledTaskDao") MongoScheduledTaskDao mongoDao,
            TwinDaoConfiguration twinDaoConfiguration)
    {
        return TwinDaoFactory.createProxyDao(
                ScheduledTaskDao.class, mongoDao, ydbDao, twinDaoConfiguration, TwinDaoInvocationHandler.EMPTY_RESULT);
    }

    @Bean
    public YdbSessionPoolMetrics ydbMetrics(TableClient tableClient) {
        return new YdbSessionPoolMetrics(tableClient);
    }
}
