package ru.yandex.chemodan.app.docviewer.dao.ydb;

import lombok.Data;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDaoUtils;
import ru.yandex.chemodan.app.docviewer.log.DocviewerTskvEvent;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.StoredImage;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbQueryMapper;
import ru.yandex.chemodan.ydb.dao.pojo.OneTablePojoYdbDao;
import ru.yandex.chemodan.ydb.dao.pojo.YdbClassAnalyzer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;


/**
 * @author yashunsky
 */
public class YdbImageDao extends OneTablePojoYdbDao<YdbImageDao.YdbStoredImage> implements ImageDao {

    public static final String COLUMN_FILE_ID = "file_id";
    public static final String COLUMN_SUB_ID = "sub_id";
    public static final String COLUMN_LAST_ACCESS = "last_access";

    @Value("${cleanup.threads}")
    private int cleanupThreads;

    protected YdbImageDao(ThreadLocalYdbTransactionManager transactionManager) {
        super(transactionManager, "images", YdbStoredImage.class, YdbClassAnalyzer
                .getDescription(YdbStoredImage.class, Cf.list(COLUMN_FILE_ID, COLUMN_SUB_ID), Cf.map()));
    }

    @Override
    public Option<StoredImage> findOne(String fileId, int oneBasedPageIndex, DimensionO size) {
        return findOne(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size));
    }

    @Override
    public Option<StoredImage> findOne(String fileId, String subKey) {
        return findOne(getSqlIdCondition(fileId, subKey)).map(YdbStoredImage::toStoredImage);
    }

    @Override
    public void updateLastAccessTime(String fileId, int oneBasedPageIndex, DimensionO size) {
        updateLastAccessTime(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size));
    }

    @Override
    public void updateLastAccessTime(String fileId, String subKey) {
        update(getYdbIdCondition(fileId, subKey), Cf.map(COLUMN_LAST_ACCESS, Instant.now()));
        DocviewerTskvEvent.imageAccessed(fileId, subKey).log();
    }

    @Override
    public void savePdfImageIfNotExists(String fileId, int oneBasedPageIndex, DimensionO size, String fileLinkString) {
        saveImageIfNotExists(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size), fileLinkString);
    }

    @Override
    public void saveImageIfNotExists(String fileId, String subKey, String fileLinkString) {
        StoredImage storedImage = new StoredImage(fileId, subKey, Instant.now(), fileLinkString);
        upsert(YdbStoredImage.fromStoredImage(storedImage));
        DocviewerTskvEvent.imageAccessed(fileId, subKey).log();
    }

    @Override
    public void delete(String fileId, int oneBasedPageIndex, DimensionO size) {
        delete(fileId, ImageDaoUtils.generateSubId(oneBasedPageIndex, size));
    }

    @Override
    public void delete(String fileId, String subKey) {
        delete(getSqlIdCondition(fileId, subKey));
        DocviewerTskvEvent.imageDeleted(fileId, subKey).log();
    }

    @Override
    public void deleteByLastAccessLessBatch(Instant timestamp, Function1V<StoredImage> deleteHandler) {
        // not implemented yet
    }

    @Override
    public void deleteByFileIdBatch(String fileId, Function1V<StoredImage> deleteHandler) {
        SqlCondition condition = getColumnCondition(COLUMN_FILE_ID, fileId);
        DvYdbUtils.forEachBatch(this, condition, COLUMN_SUB_ID,
                YdbStoredImage::getSubId, cleanupThreads, i -> deleteHandler.apply(i.toStoredImage()));
    }

    private SqlCondition getSqlIdCondition(String fileId, String subKey) {
        return getColumnCondition(COLUMN_FILE_ID, fileId).and(ConditionUtils.column(COLUMN_SUB_ID).eq(subKey));
    }

    private YdbQueryMapper.YdbCondition getYdbIdCondition(String fileId, String subKey) {
        return YdbQueryMapper.mapWhereSql(getSqlIdCondition(fileId, subKey));
    }

    @BenderBindAllFields
    @Data
    public static class YdbStoredImage {
        public final String fileId;
        public final String subId;

        public final Instant lastAccess;
        public final String fileLink;

        public StoredImage toStoredImage() {
            return new StoredImage(fileId, subId, lastAccess, fileLink);
        }

        public static YdbStoredImage fromStoredImage(StoredImage i) {
            return new YdbStoredImage(i.fileId, i.subId, i.lastAccess, i.fileLink);
        }
    }
}
