package ru.yandex.chemodan.app.docviewer.dao.ydb;

import java.util.Collection;

import com.yandex.ydb.table.transaction.TransactionMode;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupTarget;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupTask;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfWarmupMetrics;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.pojo.OneTablePojoYdbDao;
import ru.yandex.chemodan.ydb.dao.pojo.YdbClassAnalyzer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.q.SqlCondition;

/**
 * @author yashunsky
 */
public class YdbPdfWarmupDao extends OneTablePojoYdbDao<YdbPdfWarmupDao.YdbPdfWarmupTask> implements PdfWarmupDao {

    private static final String COLUMN_FILE_ID = "file_id";
    private static final String COLUMN_WIDTH = "width";
    private static final String COLUMN_HEIGHT = "height";
    private static final String COLUMN_MOBILE = "mobile";
    private static final String COLUMN_BLOCK_INDEX = "block_index";
    private static final String COLUMN_BLOCK_SIZE = "block_size";
    private static final String COLUMN_CTIME = "ctime";

    private final int blockSize;
    private final Function0<Boolean> affectMetricF;

    public YdbPdfWarmupDao(
            ThreadLocalYdbTransactionManager transactionManager,
            int blockSize, Duration ttl, Function0<Boolean> affectMetricF)
    {
        super(transactionManager, "pdf_warmup_task", YdbPdfWarmupTask.class,
                YdbClassAnalyzer.getDescription(
                        YdbPdfWarmupTask.class,
                        Cf.list(COLUMN_FILE_ID, COLUMN_WIDTH, COLUMN_HEIGHT,
                                COLUMN_MOBILE, COLUMN_BLOCK_INDEX, COLUMN_BLOCK_SIZE),
                        Cf.map(),
                        COLUMN_CTIME,
                        ttl));
        this.blockSize = blockSize;
        this.affectMetricF = affectMetricF;
    }

    @Override
    public Collection<PdfWarmupTask> createTasks(PdfWarmupTarget target, int zeroBasedStartPageIndex,
            int zeroBasedEndPageIndex)
    {
        boolean affectMetric = affectMetricF.apply();
        ListF<PdfWarmupTask> tasks = Cf.arrayList();
        int startBlockIndex = zeroBasedStartPageIndex / blockSize;
        int endBlockIndex = zeroBasedEndPageIndex / blockSize;
        for (int i = startBlockIndex; i <= endBlockIndex; i++) {
            PdfWarmupTask task = new PdfWarmupTask(target, i, blockSize);
            if (affectMetric) {
                PdfWarmupMetrics.totalBlocks.inc();
            }

            YdbPdfWarmupTask ydbTask = YdbPdfWarmupTask.fromTaskAndTime(task, Instant.now());

            boolean isNew = transactionManager.executeInTx(() -> {
                boolean exists = findOne(getPrimaryCondition(ydbTask)).isPresent();
                upsert(ydbTask);
                return !exists;
            }, TransactionMode.SERIALIZABLE_READ_WRITE);

            if (isNew) {
                if (affectMetric) {
                    PdfWarmupMetrics.warmBlocks.inc();
                }
                tasks.add(task);
            }
        }
        return tasks;
    }

    @Override
    public void cleanup(Duration ttl) {
        // YDB auto cleaning by ttl
    }

    @Override
    public void cleanup() {
        // YDB auto cleaning by ttl
    }

    public void forceCleanup(Duration ttl) {
        // YDB auto cleaning by ttl
    }

    private SqlCondition getPrimaryCondition(YdbPdfWarmupTask task) {
        return getColumnCondition(COLUMN_FILE_ID, task.fileId).and(
                getColumnCondition(COLUMN_WIDTH, task.width),
                getColumnCondition(COLUMN_HEIGHT, task.height),
                getColumnCondition(COLUMN_MOBILE, task.mobile),
                getColumnCondition(COLUMN_BLOCK_INDEX, task.blockIndex),
                getColumnCondition(COLUMN_BLOCK_SIZE, task.blockSize)
        );
    }

    @AllArgsConstructor
    @BenderBindAllFields
    @Data
    public static class YdbPdfWarmupTask {
        public final String fileId;
        public final int width;
        public final int height;
        public final boolean mobile;
        public final int blockIndex;
        public final int blockSize;
        public final Instant ctime;

        public static YdbPdfWarmupTask fromTaskAndTime(PdfWarmupTask task, Instant ctime) {
            PdfWarmupTarget target = task.target;
            return new YdbPdfWarmupTask(
                    target.fileId, target.width, target.height, target.mobile,
                    task.blockIndex, task.blockSize,
                    ctime);
        }
    }
}
