package ru.yandex.chemodan.app.docviewer.dao.ydb;

import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.values.PrimitiveValue;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.dao.results.ConvertErrorArgs;
import ru.yandex.chemodan.app.docviewer.dao.results.ConvertSuccessArgs;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbQueryMapper;
import ru.yandex.chemodan.ydb.dao.YdbRowMapper;
import ru.yandex.chemodan.ydb.dao.pojo.OneTablePojoYdbDao;
import ru.yandex.chemodan.ydb.dao.pojo.YdbClassAnalyzer;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author yashunsky
 */
public class YdbStoredResultDao extends OneTablePojoYdbDao<StoredResult> implements StoredResultDao {

    private static final String COLUMN_FILE_ID = "file_id";
    private static final String COLUMN_CONVERT_TARGET = "convert_target_type";
    private static final String COLUMN_LAST_ACCESS_TIME = "last_access";
    private static final String COLUMN_WEIGHT = "weight";
    private static final String COLUMN_DOCUMENT_PROPERTIES = "properties";
    private static final String COLUMN_RESTORE_URI = "restore_uri";
    private static final String COLUMN_REMOTE_FILE_ID = "remote_file_id";

    private static final String INDEX_REMOTE_ID = "remote_file_id_index";

    private final ThreadLocalYdbTransactionManager transactionManager;
    private final AliveAppInfo aliveAppInfo;

    public YdbStoredResultDao(ThreadLocalYdbTransactionManager transactionManager, AliveAppInfo aliveAppInfo, Duration ttl) {
        super(transactionManager, "results", StoredResult.class,
                YdbClassAnalyzer.getDescription(
                        StoredResult.class,
                        Cf.list(COLUMN_FILE_ID, COLUMN_CONVERT_TARGET),
                        Cf.map(INDEX_REMOTE_ID, Cf.list(COLUMN_REMOTE_FILE_ID)),
                        COLUMN_LAST_ACCESS_TIME,
                        ttl));
        this.transactionManager = transactionManager;
        this.aliveAppInfo = aliveAppInfo;
    }

    @Override
    public void delete(String fileId, TargetType convertTargetType) {
        delete(getSqlIdCondition(fileId, convertTargetType));
    }

    @Override
    public Option<StoredResult> find(String fileId, TargetType convertTargetType) {
        return findOne(getSqlIdCondition(fileId, convertTargetType));
    }

    @Override
    public Option<StoredResult> findAnyOfTypes(String fileId, ListF<TargetType> targetTypes) {
        return findOne(getColumnCondition(COLUMN_FILE_ID, fileId)
                .and(ConditionUtils.column(COLUMN_CONVERT_TARGET).inSet(targetTypes.map(TargetType::name))));
    }

    @Override
    public Iterable<StoredResult> findByLastAccessLess(Instant timestamp) {
        // Currently not supported
        return Cf.list();
    }

    @Override
    public void saveOrUpdateResult(ConvertSuccessArgs args) {
        StoredResult storedResult = new StoredResult();

        storedResult.setFileId(args.getFileId());
        storedResult.setConvertTargetType(args.getTargetType());

        storedResult.setLastAccess(Instant.now());
        storedResult.setFileLink(Option.of(args.getResultFileLink()));
        storedResult.setLength(Option.of(args.getLength()));
        storedResult.setPages(Option.of(args.getPages()));
        storedResult.setWeight(args.getWeight());
        storedResult.setRestoreUri(Option.of(args.getRestoreUri()));
        storedResult.setPagesInfo(args.getPagesInfo());
        storedResult.setConvertResultType(Option.of(args.getType()));
        storedResult.setPasswordHash(args.getRawPassword());
        storedResult.setRemoteFileId(args.getRemoteFileId());
        storedResult.setContentType(args.getContentType());
        storedResult.setProperties(Option.of(args.getProperties()));
        storedResult.setDocviewerVersion(Option.of(aliveAppInfo.getVersion()));

        save(storedResult);
    }

    @Override
    public void saveOrUpdateResult(ConvertErrorArgs args) {
        StoredResult storedResult = new StoredResult();

        storedResult.setFileId(args.getFileId());
        storedResult.setConvertTargetType(args.getTargetType());

        storedResult.setError(Option.of(args.getError()));
        storedResult.setErrorCode(Option.of(args.getErrorCode()));
        storedResult.setLastAccess(Instant.now());
        storedResult.setFailedAttemptsCount(Option.of(args.getFailedAttemptsCount()));
        storedResult.setPackageVersion(Option.of(args.getPackageVersion()));
        storedResult.setDocviewerVersion(Option.of(aliveAppInfo.getVersion()));

        save(storedResult);
    }

    @Override
    public void updateLastAccessTime(String fileId, TargetType targetType) {
        setField(fileId, targetType, COLUMN_LAST_ACCESS_TIME, Instant.now());
    }

    @Override
    public void updateWeight(String fileId, TargetType convertTargetType, long weight) {
        setField(fileId, convertTargetType, COLUMN_WEIGHT, weight);
    }

    @Override
    public void addExtractedText(String fileId, TargetType convertTargetType, String link) {
        YdbQueryMapper.YdbCondition condition = getYdbIdCondition(fileId, convertTargetType);
        String findSql = condition.declareSql
                + "SELECT " + COLUMN_DOCUMENT_PROPERTIES + " FROM " + getTableName() + " " + condition.whereSql;

        transactionManager.executeInTx(() -> {
            Option<DocumentProperties> current =
                    queryForList(findSql, toParams(condition.params), DocumentPropertiesMapper.M).firstO();
            if (current.isPresent()) {
                DocumentProperties patched = current.get().withProperty(DocumentProperties.EXTRACTED_TEXT, link);

                PrimitiveValue value = PrimitiveValue.json(new String(
                        DocumentProperties.S.serializeJson(patched), CharsetUtils.UTF8_CHARSET));

                update(condition, Cf.map(COLUMN_DOCUMENT_PROPERTIES, value));
            }
            return null;
        }, TransactionMode.SERIALIZABLE_READ_WRITE);
    }

    @Override
    public void setRestoreUri(String fileId, TargetType convertTargetType, String restoreUri) {
        setField(fileId, convertTargetType, COLUMN_RESTORE_URI, restoreUri);
    }

    @Override
    public void deleteByLastAccessLessBatch(Instant timestamp, Function1V<StoredResult> deleteHandler) {
        // not implemented yet
    }

    @Override
    public void handleEachResult(Function1V<StoredResult> handler) {
        //useless in prod
    }

    @Override
    public ListF<StoredResult> findByRemoteId(String remoteId) {
        return find(getColumnCondition(COLUMN_REMOTE_FILE_ID, remoteId), INDEX_REMOTE_ID);
    }

    private void setField(String fileId, TargetType convertTargetType, String columnName, Object value) {
        setFields(fileId, convertTargetType, Cf.map(columnName, value));
    }

    private void setFields(String fileId, TargetType convertTargetType, MapF<String, Object> toUpdate) {
        update(getYdbIdCondition(fileId, convertTargetType), toUpdate);
    }

    private SqlCondition getSqlIdCondition(String fileId, TargetType targetType) {
        return getColumnCondition(COLUMN_FILE_ID, fileId)
                .and(ConditionUtils.column(COLUMN_CONVERT_TARGET).eq(targetType.toString()));
    }

    private YdbQueryMapper.YdbCondition getYdbIdCondition(String fileId, TargetType targetType) {
        return YdbQueryMapper.mapWhereSql(getSqlIdCondition(fileId, targetType));
    }

    public void save(StoredResult storedResult) {
        upsert(storedResult);
    }

    private static class DocumentPropertiesMapper implements YdbRowMapper<DocumentProperties> {
        public final static DocumentPropertiesMapper M = new DocumentPropertiesMapper();

        @Override
        public DocumentProperties mapRow(ResultSetReader rs, int rowNum) {
            return DocumentProperties.P.parseJson(rs.getColumn(COLUMN_DOCUMENT_PROPERTIES).getOptionalItem().getJson());
        }
    }
}
