package ru.yandex.chemodan.app.docviewer.dao.ydb;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbQueryMapper;
import ru.yandex.chemodan.ydb.dao.pojo.OneTablePojoYdbDao;
import ru.yandex.chemodan.ydb.dao.pojo.YdbClassAnalyzer;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
public class YdbStoredUriDao extends OneTablePojoYdbDao<StoredUri> implements StoredUriDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbStoredUriDao.class);

    private static final String COLUMN_URI = "uri";
    private static final String COLUMN_PASSWORDS = "passwords";
    private static final String COLUMN_CONTENT_TYPE = "content_type";
    private static final String COLUMN_TIMESTAMP = "timestamp";
    private static final String COLUMN_CONVERT_TARGETS = "convert_targets";
    private static final String COLUMN_FILE_ID = "file_id";
    private static final String COLUMN_CONTENT_SIZE = "content_size";
    private static final String COLUMN_SERP_LAST_ACCESS = "serp_last_access";
    private static final String COLUMN_ERROR = "error";
    private static final String COLUMN_ERROR_CODE = "error_code";
    private static final String COLUMN_ERROR_ARCHIVE_PATH = "error_archive_path";
    private static final String COLUMN_MAX_CONTENT_SIZE = "max_content_size";

    private static final String INDEX_FILE_ID = "file_id_index";


    public YdbStoredUriDao(ThreadLocalYdbTransactionManager transactionManager, Duration ttl) {
        super(transactionManager, "uris", StoredUri.class, YdbClassAnalyzer.getDescription(
                StoredUri.class,
                Cf.list(COLUMN_URI),
                Cf.map(INDEX_FILE_ID, Cf.list(COLUMN_FILE_ID)),
                COLUMN_TIMESTAMP,
                ttl));
    }

    @Override
    public void delete(ActualUri uri) {
        delete(getSqlIdCondition(uri));
    }

    @Override
    public Option<StoredUri> find(ActualUri uri) {
        return findOne(getSqlIdCondition(uri));
    }

    @Override
    public ListF<StoredUri> findAllByFileId(String fileId) {
        return find(getColumnCondition(COLUMN_FILE_ID, fileId), INDEX_FILE_ID);
    }

    @Override
    public Tuple2List<String, String> getPasswordsSorted(ActualUri uri) {
        return find(uri).get().getPasswords();
    }

    @Override
    public Option<Tuple2List<String, String>> getPasswordsSortedO(ActualUri uri) {
        return find(uri).map(StoredUri::getPasswords);
    }

    @Override
    public void updatePasswords(ActualUri uri, Tuple2List<String, String> passwords) {
        logger.debug("Setting passwords of URI '{}'", uri);
        StoredUri toUpdate = new StoredUri();
        toUpdate.setPasswords(passwords);
        update(getYdbIdCondition(uri), toUpdate, Cf.set(COLUMN_PASSWORDS));
    }

    @Override
    public void updatePasswordRaw(ActualUri uri, String archivePath, String rawValue) {
        Option<Tuple2List<String, String>> passwordsO = getPasswordsSortedO(uri);
        if (!passwordsO.isPresent()) {
            logger.debug("URI '{}' not found, don't update password for path '{}'", uri, archivePath);
            return;
        }

        Tuple2List<String, String> passwords = passwordsO.get();
        String newHashValue = SessionKey.toHashValue(rawValue);

        Tuple2<Tuple2List<String, String>, Tuple2List<String, String>> partition =
                passwords.partitionBy1(Cf.String.equalsF(archivePath));

        if (partition.get1().isNotEmpty()) {
            if (partition.get1().single().get2().equals(newHashValue)) {
                return;
            }
            passwords = partition.get2();
        }
        passwords = passwords.plus1(archivePath, newHashValue);

        updatePasswords(uri, passwords);

        logger.debug("Password updated for URI '{}' and path '{}'", uri, archivePath);
    }

    @Override
    public void saveOrUpdateUri(ActualUri uri, Option<String> contentType, TargetType convertTargetType,
            Float priority)
    {
        logger.debug("Creating URI '{}' with target '{}' and priority '{}'", uri, convertTargetType, priority);

        YdbQueryMapper.YdbCondition condition = getYdbIdCondition(uri);

        saveOrUpdate(condition,
                () -> {
                    StoredUri toCreate = new StoredUri();
                    toCreate.setUri(uri);
                    toCreate.setTimestamp(Instant.now());
                    toCreate.setContentType(contentType);
                    toCreate.setConvertTargets(Cf.map(convertTargetType, priority));
                    return toCreate;
                },
                storedUri -> {
                    storedUri.setTimestamp(Instant.now());
                    contentType.ifPresent(storedUri::setContentType);
                    storedUri.setConvertTargets(storedUri.getConvertTargets().plus1(convertTargetType, priority));
                    return storedUri;
                },
                Cf.set(COLUMN_TIMESTAMP, COLUMN_CONTENT_TYPE, COLUMN_CONVERT_TARGETS));
    }

    @Override
    public void updateUri(
            ActualUri uri, Option<String> reportedContentType,
            String fileId, DataSize size, Option<Instant> serpLastAccess)
    {
        logger.debug("Setting content type and file ID of URI '{}' to '{}' and '{}'",
                uri, reportedContentType, fileId);

        MapF<String, Object> toUpdate = Cf.hashMap();

        toUpdate.put(COLUMN_FILE_ID, fileId);
        toUpdate.put(COLUMN_CONTENT_SIZE, size.toBytes());

        serpLastAccess.ifPresent(sla -> toUpdate.put(COLUMN_SERP_LAST_ACCESS, sla));
        reportedContentType.ifPresent(rct -> toUpdate.put(COLUMN_CONTENT_TYPE, rct));

        setFields(uri, toUpdate);
    }

    @Override
    public void updateUriClean(ActualUri uri) {
        logger.debug("Clearing copy result information about '{}'", uri);

        unset(getYdbIdCondition(uri), Cf.set(
                COLUMN_CONTENT_TYPE, COLUMN_FILE_ID, COLUMN_ERROR, COLUMN_ERROR_CODE, COLUMN_ERROR_ARCHIVE_PATH));
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp, Function1V<StoredUri> deleteHandler) {
        // not implemented yet
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp) {
        // YDB auto cleaning by ttl
    }

    @Override
    public void deleteErrorsByTimestampLessBatch(Instant timestamp) {
        // YDB auto cleaning by ttl
    }

    @Override
    public void updateUri(ActualUri uri, ErrorCode errorCode, String stackTrace, Option<String> errorArchivePath,
            Option<Long> contentSize, Option<Long> maxSize)
    {
        logger.debug("Setting error code and text of URI '{}' to '{}' and '{}'", uri, errorCode, stackTrace);

        MapF<String, Object> toUpdate = Cf.hashMap();
        toUpdate.put(COLUMN_ERROR, stackTrace);
        toUpdate.put(COLUMN_ERROR_CODE, errorCode.name());

        errorArchivePath.ifPresent(eap -> toUpdate.put(COLUMN_ERROR_ARCHIVE_PATH, eap));
        contentSize.ifPresent(cs -> toUpdate.put(COLUMN_CONTENT_SIZE, cs));
        maxSize.ifPresent(ms -> toUpdate.put(COLUMN_MAX_CONTENT_SIZE, ms));

        setFields(uri, toUpdate);
    }

    private SqlCondition getSqlIdCondition(ActualUri uri) {
        return getColumnCondition(COLUMN_URI, uri.getUriString());
    }

    private YdbQueryMapper.YdbCondition getYdbIdCondition(ActualUri uri) {
        return YdbQueryMapper.mapWhereSql(getSqlIdCondition(uri));
    }

    private void setFields(ActualUri uri, MapF<String, Object> toUpdate) {
        update(getYdbIdCondition(uri), toUpdate);
    }
}
