package ru.yandex.chemodan.app.docviewer.dao.ydb;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbQueryMapper;
import ru.yandex.chemodan.ydb.dao.pojo.OneTablePojoYdbDao;
import ru.yandex.chemodan.ydb.dao.pojo.YdbClassAnalyzer;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;

/**
 * @author yashunsky
 */
public class YdbUriRightsDao extends OneTablePojoYdbDao<YdbUriRightsDao.UriRights> implements UriRightsDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbUriRightsDao.class);

    private static final String COLUMN_URI = "uri";
    private static final String COLUMN_UID = "uid";
    private static final String COLUMN_TIMESTAMP = "timestamp";
    private static final String COLUMN_FILE_ID = "file_id";

    private static final String INDEX_UID_FILE_ID = "uid_file_id_index";

    private static final Duration ACCESS_TIME_PRECISION = Duration.standardMinutes(5);

    public YdbUriRightsDao(ThreadLocalYdbTransactionManager transactionManager, Duration ttl) {
        super(transactionManager, "rights", UriRights.class,
                YdbClassAnalyzer.getDescription(
                        UriRights.class,
                        Cf.list(COLUMN_URI, COLUMN_UID),
                        Cf.map(INDEX_UID_FILE_ID, Cf.list(COLUMN_UID, COLUMN_FILE_ID)),
                        COLUMN_TIMESTAMP,
                        ttl));
    }

    @Override
    public void deleteUriRights(ActualUri uri) {
        try {
            delete(getSqlUriCondition(uri));
            LoggerEventsRecorder.saveCleanupUriEvent(getTableName(), uri);
        } catch (Exception exc) {
            logger.error("Unable to delete URL rights for '" + uri + "': " + exc, exc);
            LoggerEventsRecorder.saveCleanupUriFailedEvent(getTableName(), exc, uri);
        }
    }

    @Override
    public boolean findExistsUriRight(ActualUri uri, PassportUidOrZero uid) {
        return findOne(getSqlIdCondition(uri, uid)).isPresent();
    }

    @Override
    public Option<ActualUri> findUriByFileIdAndUid(String fileId, PassportUidOrZero uid) {
        return findOne(getSqlUidCondition(uid).and(getSqlFileIdCondition(fileId)),
                Option.of(INDEX_UID_FILE_ID),
                SqlOrder.orderByColumnDesc(COLUMN_TIMESTAMP), SqlLimits.first(1)).map(UriRights::getUri);
    }

    @Override
    public ListF<ActualUri> findUrisAccessedByUid(PassportUidOrZero uid) {
        return find(getSqlUidCondition(uid), Option.of(INDEX_UID_FILE_ID)).map(UriRights::getUri);
    }

    @Override
    public void saveOrUpdateUriRight(ActualUri uri, PassportUidOrZero uid) {
        Instant updateEdge = Instant.now().minus(ACCESS_TIME_PRECISION);
        boolean newEnough = findOne(getSqlIdCondition(uri, uid))
                .map(UriRights::getTimestamp)
                .exists(lastAccessTime -> lastAccessTime.isAfter(updateEdge));
        if (newEnough) {
            logger.info("Access to URL '{}' for '{}' already new enough", uri, uid);
            return;
        }

        UriRights uriRights = new UriRights();
        uriRights.setUri(uri);
        uriRights.setUid(uid.getUid());
        uriRights.setTimestamp(Instant.now());

        upsert(uriRights);

        logger.info("Access to URL '{}' for '{}' stored", uri, uid);
    }

    @Override
    public void updateUriRights(ActualUri uri, String fileId) {
        YdbQueryMapper.YdbCondition condition = YdbQueryMapper.mapWhereSql(
                getSqlUriCondition(uri).and(
                        SqlCondition.column(COLUMN_FILE_ID).ne(fileId)
                                .or(SqlCondition.column(COLUMN_FILE_ID).isNull()))
        );
        update(condition, Cf.map(COLUMN_FILE_ID, fileId));
    }

    @Override
    public boolean validate(String fileId, PassportUidOrZero uid) {
        return findOne(getSqlUidCondition(uid).and(getSqlFileIdCondition(fileId)), INDEX_UID_FILE_ID).isPresent();
    }

    @Override
    public void deleteByTimestampLessBatch(Instant timestamp) {
        // YDB auto cleaning by ttl
    }

    private void delete(UriRights uriRights) {
        delete(getSqlIdCondition(uriRights.getUri(), PassportUidOrZero.fromUid(uriRights.getUid())));
    }

    private SqlCondition getSqlUriCondition(ActualUri uri) {
        return getColumnCondition(COLUMN_URI, uri.getUriString());
    }

    private SqlCondition getSqlUidCondition(PassportUidOrZero uid) {
        return getColumnCondition(COLUMN_UID, uid.getUid());
    }

    private SqlCondition getSqlFileIdCondition(String fileId) {
        return getColumnCondition(COLUMN_FILE_ID, fileId);
    }

    private SqlCondition getSqlIdCondition(ActualUri uri, PassportUidOrZero uid) {
        return getSqlUriCondition(uri).and(getSqlUidCondition(uid));
    }

    @Data
    @BenderBindAllFields
    public static class UriRights {
        private ActualUri uri;
        private long uid;
        private Option<String> fileId = Option.empty();
        private Instant timestamp;
    }
}
