package ru.yandex.chemodan.app.docviewer.disk;

import java.io.IOException;

import lombok.val;
import org.apache.commons.lang3.Validate;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsOperationStatus;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsOperationStatusWithMeta;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsUrlHelper;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsUtils;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskPrivateFileId;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskPublicFileId;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceType;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.MpfsHttpClient;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.InternalServerErrorException;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.docviewer.web.backend.SourceInternalAction.buildDownloadFileUUID;

/**
 * @author akirakozov
 */
public class DiskManager {
    private static final String DOCVIEWER_SERVICE = "docviewer";
    private static final Logger logger = LoggerFactory.getLogger(DiskManager.class);

    @Autowired
    private MpfsUrlHelper mpfsUrlHelper;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private MpfsHttpClient mpfsHttpClient;

    @Value("${serp.download.url}")
    private String serpDownloadUrl;

    public void initUser(PassportUidOrZero uid, Option<String> locale, boolean publish) {
        String source = publish ? MpfsUrlHelper.SOURCE_DV_SHARE : MpfsUrlHelper.SOURCE_COPY_DV;
        String url = mpfsUrlHelper.getUserInitUrl(uid, locale, source);
        executeReadString(new HttpGet(url));
    }

    public Option<DiskResourceId> getDiskResourceId(DocumentSourceInfo source, ActualUri uri) {
        return uriHelper.getDiskResourceId(source, uri);
    }

    public Option<DiskResourceId> getDiskResourceId(DocumentSourceInfo source) {
        return uriHelper.getDiskResourceId(source, uriHelper.rewrite(source));
    }

    public String copyResourceToDisk(PassportUidOrZero uid, DiskResourceId resourceId,
            Option<String> fileName, boolean publish, boolean showNda)
    {
        Validate.isTrue(uid.isAuthenticated());

        String copyToDiskUrl = resourceId.getCopyToDiskUrl(mpfsUrlHelper, uid, fileName, publish);

        copyToDiskUrl = mpfsUrlHelper.addUidIfMissing(copyToDiskUrl, uid);

        return executeReadString(new HttpGet(copyToDiskUrl));
    }

    public String asyncCopyToDiskAndGetOperationId(PassportUidOrZero uid,
            DiskResourceId resourceId, Option<String> fileName, boolean publish, boolean showNda)
    {
        Validate.isTrue(resourceId.getType().isAsyncCopySupported(),
                "async copy is supported only for mail attachments and web documents");

        if (resourceId.getType() == DiskResourceType.MAIL_ATTACHMENT) {
            MpfsUtils.MpfsAsyncCopyResponse resp = MpfsUtils.parseMpfsAsyncCopy(
                    copyResourceToDisk(uid, resourceId, fileName, publish, showNda));
            Validate.isTrue(resp.status == 1, "Unsuccess response on async-copy");
            return resp.result.oid;
        } else {
            MpfsUtils.MpfsAsyncStoreExternalResponse resp = MpfsUtils.parseMpfsAsyncStoreExternal(
                    copyResourceToDisk(uid, resourceId, fileName, publish, showNda));
            return resp.oid;
        }
    }

    private String importFileFromArchiveWithDocviewer(PassportUidOrZero uid, DocumentSourceInfo documentSourceInfo,
            Option<String> fileName)
    {
        Option<String> archivePath = documentSourceInfo.getArchivePath()
                .map(path -> StringUtils.startsWith(path, "//") ? path : "//" + path);

        String serviceFileId = uid.toString() + ":" + buildDownloadFileUUID(documentSourceInfo.getOriginalUrl(),
                archivePath);
        String importFromServiceUrl = mpfsUrlHelper.getAsyncImportFileFromServiceUrl(
                uid, DOCVIEWER_SERVICE, serviceFileId, fileName.get());

        return executeReadString(new HttpGet(importFromServiceUrl));
    }

    public String extractAndSaveFileFromArchiveToDiskAndGetOperationId(PassportUidOrZero uid,
            DocumentSourceInfo documentSourceInfo, Option<String> fileName)
    {
        Validate.isTrue(uid.isAuthenticated());

        String responseString = importFileFromArchiveWithDocviewer(uid, documentSourceInfo, fileName);

        return MpfsUtils.parseMpfsExtractFileFromArchive(responseString).oid;
    }

    public String shareResourceAndGetShortUrl(PassportUidOrZero uid, DiskResourceId resourceId) {
        switch (resourceId.getType()) {
            case DISK_PUBLIC_FILE:
                return getPublicFileShortUrl(uid, (DiskPublicFileId) resourceId);
            case DISK_PRIVATE_FILE:
                return setPublicAndGetShortUrl(uid, (DiskPrivateFileId) resourceId);
            default:
                throw new IllegalArgumentException("Couldn't share resource with type" + resourceId.getType());
        }
    }

    public Option<String> getHid(PassportUidOrZero uid, DiskResourceId resourceId) {

        switch (resourceId.getType()) {
            case DISK_PUBLIC_FILE:
                return getPublicFileHid(uid, (DiskPublicFileId) resourceId);
            case DISK_PRIVATE_FILE:
                return getFileHid(uid, (DiskPrivateFileId) resourceId);
            default:
                throw new IllegalArgumentException("Couldn't get hid for not disk type: " + resourceId.getType());
        }
    }

    private Option<String> getFileHid(PassportUidOrZero uid, DiskPrivateFileId resourceId) {
        String url = mpfsUrlHelper.getInfoUrl(uid, resourceId.getPath(), Cf.set("hid"));
        String resp = executeReadString(new HttpGet(url));
        return MpfsUtils.getHidFromInfo(resp);
    }

    private Option<String> getPublicFileHid(PassportUidOrZero uid, DiskPublicFileId id) {
        String url = mpfsUrlHelper.getPublicInfoUrl(uid, id.getPrivateHash(), Cf.set("hid"));
        String resp = executeReadString(new HttpGet(url));
        return MpfsUtils.getHidFromPublicInfo(resp);
    }

    public boolean isMpfsHost(ActualUri uri) {
        return mpfsUrlHelper.getMpfsHost().equals(uri.getUri().getHost());
    }

    public MpfsOperationStatusWithMeta getMpfsOperationStatusWithMeta(PassportUidOrZero uid, String oid,
            Option<String> meta)
    {
        SetF<String> metaSet = meta.isPresent() ? Cf.set(meta.get()) : Cf.set();
        String url = mpfsUrlHelper.getOperationStatusUrl(uid, oid, metaSet);
        String resp = executeReadString(new HttpGet(url));
        MpfsOperationStatus status = MpfsUtils.getOperationStatus(resp);

        if (status == MpfsOperationStatus.DONE && meta.isPresent()) {
            return new MpfsOperationStatusWithMeta(status, Option.of(MpfsUtils.getCopiedResourceMetaFromStatus(resp)));
        } else {
            return new MpfsOperationStatusWithMeta(status);
        }
    }

    public AsyncCopyStatus getAsyncCopyStatus(PassportUidOrZero uid, String oid) {
        String url = mpfsUrlHelper.getOperationStatusUrl(uid, oid, Cf.set("short_url"));
        String resp = executeReadString(new HttpGet(url));
        MpfsOperationStatus status = MpfsUtils.getOperationStatus(resp);
        if (status == MpfsOperationStatus.DONE) {
            return new AsyncCopyStatus(status, Option.of(MpfsUtils.getCopiedResourceShortUrlFromStatus(resp)));
        } else {
            return new AsyncCopyStatus(status);
        }
    }

    public String getPublicFileShortUrl(PassportUidOrZero uid, DiskPublicFileId id) {
        String url = mpfsUrlHelper.getPublicInfoUrl(uid, id.getPrivateHash(), Cf.set("short_url"));
        String resp = executeReadString(new HttpGet(url));
        return MpfsUtils.getShortUrlFromPublicInfo(resp);
    }

    public String getFileShortUrl(PassportUidOrZero uid, String path) {
        String url = mpfsUrlHelper.getInfoUrl(uid, path, Cf.set("short_url"));
        String resp = executeReadString(new HttpGet(url));
        return MpfsUtils.getShortUrlFromInfo(resp);
    }

    public Option<String> getDefaultDownloadsFolder(PassportUidOrZero uid) {
        String url = mpfsUrlHelper.getDefaultFoldersUrl(uid);
        try {
            String resp = executeReadString(new HttpGet(url));
            return Option.of(MpfsUtils.getDownloadsFolderFromDefaultFolders(resp));
        } catch (HttpException e) {
            if (e.getStatusCode().isPresent() && HttpStatus.is4xx(e.getStatusCode().get())) {
                return Option.empty();
            } else {
                throw e;
            }
        }
    }

    public Option<MpfsUtils.MpfsOfficeActionCheckResponse> getOfficeRedactorUrl(
            PassportUidOrZero uid, DiskResourceId id, Option<String> fileName, Option<DataSize> size,
            Option<String> yandexuid)
    {
        String url = mpfsUrlHelper.getOfficeCheckActionUrl(uid, id.getService(), id.getServiceFileId(),
                fileName, size);
        try {
            HttpGet httpGet = new HttpGet(url);
            if (yandexuid.isPresent()) {
                // yandexuid cookie used only for documents from browser mds,
                // it's used for sign browser file url
                httpGet.addHeader("Cookie", "yandexuid=" + yandexuid.get());
            }
            String resp = executeReadString(httpGet);
            return Option.of(MpfsUtils.getOfficeActionCheckResponse(resp));
        } catch (HttpException e) {
            if (e.getStatusCode().isMatch(HttpStatus::is4xx)) {
                return Option.empty();
            }
            try {
                if (e.getResponseBody()
                        .map(b -> JsonParser.getInstance().parseObject(b))
                        .flatMapO(j -> j.getByPathO("title"))
                        .filterByType(JsonString.class)
                        .cast(JsonString.class)
                        .map(JsonString::getString)
                        .isMatch("OfficeFileIsReadOnlyError"::equalsIgnoreCase))
                {
                    return Option.empty();
                }
            } catch (Exception e1) {
                // fallback
            }
            throw e;
        }
    }

    public String getDiskDownloadUrl(ActualUri uri, boolean dispositionInline, PassportUidOrZero uid) {
        String mpfsUrl = mpfsUrlHelper.rewriteDirectUrlToUrl(uri.getUriString(), dispositionInline, uid);
        String result = executeReadString(new HttpGet(mpfsUrl));
        MpfsUtils.MpfsDownloadUrlResponse response = MpfsUtils.parseMpfsDownloadUrl(result);
        if (response.status == 1) {
            return response.result.file;
        } else {
            throw new InternalServerErrorException("Couldn't get download url from disk");
        }
    }

    public Option<String> getPreviewUrl(String path, PassportUidOrZero uid, String size, DiskResourceType type) {
        String mpfsUrl = resolveUrl(uid, path, size, type);
        String result = executeReadString(new HttpGet(mpfsUrl));
        return MpfsUtils.getPreviewUrlFromInfo(result);
    }

    /**
     * @param stid
     * @param fileName
     * @param uid
     * @return
     */
    public String generateZaberunUrl(String stid, PassportUidOrZero uid, String fileName, String size,
            String mediaType)
    {
        String mpfsUrl = mpfsUrlHelper.getGenerateZaberunUrl(uid, stid, Option.of(size), fileName, mediaType);
        String result = executeReadString(new HttpGet(mpfsUrl));
        return MpfsUtils.getZaberunUrl(result);
    }

    private String resolveUrl(PassportUidOrZero uid, String path, String size, DiskResourceType type) {
        switch (type) {
            case DISK_PUBLIC_FILE:
                return mpfsUrlHelper.getPublicCustomPreviewUrl(uid, path, size);
            case DISK_PRIVATE_FILE:
                return mpfsUrlHelper.getCustomPreviewUrl(uid, path, size);
            default:
                throw new IllegalArgumentException("Doesn't support " + type);
        }
    }

    String setPublicAndGetShortUrl(PassportUidOrZero uid, DiskPrivateFileId id) {
        String url = mpfsUrlHelper.getSetPublicFileUrl(uid, id.getPath());
        String resp = executeReadString(new HttpGet(url));
        return MpfsUtils.getShortUrlFromSetPublicResponse(resp);
    }

    private String executeReadString(HttpUriRequest request) {
        return RetryUtils.retryOrThrow(logger, 1,
                () -> ApacheHttpClientUtils.execute(request, mpfsHttpClient, new ReadStringResponseHandler()));
    }

    //TODO: handle 404
    public static class ReadStringResponseHandler implements ResponseHandler<String> {

        @Override
        public String handleResponse(HttpResponse response) {
            Option<String> body = Option.ofNullable(response.getEntity()).map(e -> {
                try {
                    return InputStreamSourceUtils.wrap(e.getContent()).readText("utf-8");
                } catch (IOException e1) {
                    throw new RuntimeException(e1);
                }
            });
            val code = response.getStatusLine().getStatusCode();
            if (HttpStatus.is2xx(code)) {
                return body.getOrElse("");
            } else {
                throw new HttpException(code, body.getOrNull(),
                        response.getStatusLine() + "; response= " + body.getOrElse(""));
            }
        }
    } //~
}
