package ru.yandex.chemodan.app.docviewer.disk.mpfs;

import lombok.Data;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.docviewer.disk.resource.Service;
import ru.yandex.chemodan.app.docviewer.disk.resource.SystemDirectoryType;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * http://wiki.yandex-team.ru/pochta/ya.disk/MPFS/api
 *
 * @author akirakozov
 */
@Data
public class MpfsUrlHelper {
    public static final String SOURCE_COPY_DV = "copy-dv";
    public static final String SOURCE_DV_SHARE = "copy-dv-share";
    public static final String SOURCE_DV_EDIT = "copy-dv-edit";

    @Value("${mpfs.host}")
    private String mpfsHost;

    private String getCommonPrefix() {
        return "https://" + mpfsHost + "/service/";
    }

    private String getCommonJsonPrefix() {
        return "https://" + mpfsHost + "/json/";
    }

    private String getPublicUrlPrefix(boolean direct) {
        return getCommonPrefix() + (direct ? "public_direct_url" : "public_url");
    }

    public String getGenerateZaberunUrl(PassportUidOrZero uid, String stid, Option<String> size, String fileName,
            String mediaType)
    {
        String url = UrlUtils.addParameter(getCommonJsonPrefix() + "generate_zaberun_url",
                "stid", stid,
                "content_type", mediaType,
                "file_name", fileName);
        url = setEternal(UrlUtils.addParameter(url, "inline", "true", "url_type", "preview"));
        if (uid.isAuthenticated()) {
            url = UrlUtils.addParameter(url, "uid", uid.getUid());
        }
        if (size.isPresent()) {
            url = UrlUtils.addParameter(url, "size", size.get());
        }
        return url;
    }

    private String getUrlPrefix(boolean direct) {
        return getCommonPrefix() + (direct ? "direct_url" : "url");
    }

    private String getPublicUrlInner(String hash, boolean direct) {
        return UrlUtils.addParameter(getPublicUrlPrefix(direct), "private_hash", hash);
    }

    private String getUrlInner(PassportUidOrZero uid, String path, boolean direct) {
        return UrlUtils.addParameter(getUrlPrefix(direct), "uid", Long.toString(uid.getUid()), "path", path);
    }

    public String getPublicDirectUrl(String hash) {
        return getPublicUrlInner(hash, true);
    }

    public String getPublicUrl(String hash) {
        return getPublicUrlInner(hash, false);
    }

    public String getDirectUrl(PassportUidOrZero uid, String path) {
        return getUrlInner(uid, path, true);
    }

    public String getVersioningUrl(PassportUidOrZero uid, String path, String version) {
        return UrlUtils.addParameter(getCommonPrefix() + "dv_data", "uid", Long.toString(uid.getUid()),"version_id", version, "path", path);
    }

    public String getUrl(PassportUidOrZero uid, String path) {
        return getUrlInner(uid, path, false);
    }

    public String rewriteDirectUrlToUrl(
            String directUrl, boolean dispositionInline, PassportUidOrZero uid)
    {
        String url;
        if (directUrl.startsWith(getUrlPrefix(true))) {
            url = getUrlPrefix(false) + StringUtils.substringAfter(directUrl, getUrlPrefix(true));
        } else if (directUrl.startsWith(getPublicUrlPrefix(true))) {
            url = addUidIfMissing(
                    getPublicUrlPrefix(false) + StringUtils.substringAfter(directUrl, getPublicUrlPrefix(true)), uid);
        } else {
            throw new IllegalArgumentException("Incorrect direct mpfs url: " + directUrl);
        }

        if (dispositionInline) {
            url = UrlUtils.addParameter(url, "inline", "1");
        }

        return url;
    }

    public String getAsyncCopyDefaultUrl(PassportUidOrZero uid, String path,
            Option<String> customName, boolean publish)
    {
        String url = UrlUtils.addParameter(getCommonPrefix() + "async_copy_default",
                "uid", Long.toString(uid.getUid()), "path", path);
        if (customName.isPresent()) {
            url = UrlUtils.addParameter(url, "name", customName.get());
        }
        if (publish) {
            url = UrlUtils.addParameter(url, "public", 1);
        }

        return url;
    }

    public String getPublicCopyUrl(PassportUidOrZero uid, String hash, Option<String> customName) {
        String url = UrlUtils.addParameter(getCommonPrefix() + "public_copy",
                "uid", Long.toString(uid.getUid()), "private_hash", hash);
        if (customName.isPresent()) {
            url = UrlUtils.addParameter(url, "name", customName.get());
        }

        return url;
    }

    public String getUserInitUrl(PassportUidOrZero uid, Option<String> locale, String source) {
        String url = UrlUtils.addParameter(getCommonPrefix() + "user_init",
                "uid", Long.toString(uid.getUid()),
                "source", source);
        if (locale.isPresent()) {
            url = UrlUtils.addParameter(url, "locale", locale.get());
        }

        return url;
    }

    public String getPublicInfoUrl(PassportUidOrZero uid, String hash, SetF<String> meta) {
        String url = UrlUtils.addParameter(getCommonJsonPrefix() + "public_info", "private_hash", hash);
        if (meta.isNotEmpty()) {
            url = UrlUtils.addParameter(url, "meta", StringUtils.join(meta, ","));
        }

        return addUidIfMissing(url, uid);
    }

    public String getInfoUrl(PassportUidOrZero uid, String path, SetF<String> meta) {
        String url = UrlUtils.addParameter(getCommonJsonPrefix() + "info", "uid", uid.getUid(), "path", path);
        if (meta.isNotEmpty()) {
            url = UrlUtils.addParameter(url, "meta", StringUtils.join(meta, ","));
        }

        return url;
    }

    public String getCustomPreviewUrl(PassportUidOrZero uid, String path, String size) {
        return setEternal(UrlUtils.addParameter(getInfoUrl(uid, path, Cf.set("custom_preview")), "preview_size", size));
    }

    public String getPublicCustomPreviewUrl(PassportUidOrZero uid, String path, String size) {
        return setEternal(UrlUtils.addParameter(getPublicInfoUrl(uid, path, Cf.set("custom_preview")), "preview_size", size));
    }

    public String setEternal(String url) {
        return UrlUtils.addParameter(url,"eternal", 1);
    }

    public String getSetPublicFileUrl(PassportUidOrZero uid, String path) {
        return addUidIfMissing(UrlUtils.addParameter(getCommonJsonPrefix() + "set_public",
                "uid", Long.toString(uid.getUid()), "path", path),
                uid);

    }

    public String getOperationStatusUrl(PassportUidOrZero uid, String oid, SetF<String> meta) {
        String url = UrlUtils.addParameter(getCommonJsonPrefix() + "status",
                "uid", Long.toString(uid.getUid()), "oid", oid);
        if (meta.isNotEmpty()) {
            url = UrlUtils.addParameter(url, "meta", StringUtils.join(meta, ","));
        }

        return url;
    }

    public String getDefaultFoldersUrl(PassportUidOrZero uid) {
        return UrlUtils.addParameter(getCommonJsonPrefix() + "default_folders", "uid", Long.toString(uid.getUid()));
    }

    public String getMakeSystemDirectoryUrl(PassportUidOrZero uid, SystemDirectoryType type) {
        return UrlUtils.addParameter(getCommonJsonPrefix() + "mksysdir",
                "uid", Long.toString(uid.getUid()), "type", type.toString().toLowerCase());
    }

    public String getAsyncStoreExternalUrl(PassportUidOrZero uid, String path, String externalUrl) {
        return UrlUtils.addParameter(getCommonJsonPrefix() + "async_store_external",
                "uid", Long.toString(uid.getUid()), "path", path, "external_url", externalUrl);
    }

    public String getAsyncImportFileFromServiceUrl(PassportUidOrZero uid, String service,
            String serviceFileId, String fileName)
    {
        return UrlUtils.addParameters(getCommonJsonPrefix() + "import_file_from_service",
                Cf.<String, String>map()
                        .plus1("uid", Long.toString(uid.getUid()))
                        .plus1("service_id", service)
                        .plus1("service_file_id", serviceFileId)
                        .plus1("file_name", fileName));
    }

    public String getRemoveResourceUrl(PassportUidOrZero uid, String path) {
        return UrlUtils.addParameter(getCommonJsonPrefix() + "rm", "uid", Long.toString(uid.getUid()), "path", path);
    }

    public String getOfficeCheckActionUrl(
            PassportUidOrZero uid, Service service, String serviceFileId,
            Option<String> fileName, Option<DataSize> size)
    {
        MapF<String, Object> params = Cf.<String, Object>map("uid", uid.getUid())
                        .plus1("service_id", service.getServiceName())
                        .plus1("service_file_id", serviceFileId)
                        .plus1("action", "edit")
                        .plus1("source", SOURCE_DV_EDIT);
        if (fileName.isPresent()) {
            params = params.plus1("file_name", fileName.get());
        }
        if (size.isPresent()) {
            params = params.plus1("size", size.get().toBytes());
        }
        return UrlUtils.addParameters(getCommonJsonPrefix() + "office_action_check", params);
    }

    public String addUidIfMissing(String url, PassportUidOrZero uid) {
        if (PassportUidOrZero.zero().equalsTs(uid)) {
            return url;
        }
        if (!UrlUtils.getQueryParameterFromUrl(url, "uid").isPresent()) {
            url = UrlUtils.addParameter(url, "uid", Long.toString(uid.getUid()));
        }
        return url;
    }

    public String getMpfsHost() {
        return mpfsHost;
    }

    public void setMpfsHost(String mpfsHost) {
        this.mpfsHost = mpfsHost;
    }

}
