package ru.yandex.chemodan.app.docviewer.log;

import lombok.Data;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.misc.time.MoscowTime;

@Data
public class DocviewerTskvEvent {

    private static final DateTimeFormatter lastAccessDateFormatter = DateTimeFormat.forPattern("yyyy-MM-dd");

    public static DocviewerTskvEvent documentDeleted(String fileId, TargetType targetType) {
        return new DocviewerTskvEvent(ActionEventType.DOCUMENT_DELETED, Option.of(fileId), Option.of(targetType),
                Option.empty(), Option.empty(), Option.empty());
    }

    public static DocviewerTskvEvent documentSkipped(String fileId, TargetType targetType, long weight, Instant lastAccessTime) {
        return new DocviewerTskvEvent(ActionEventType.DOCUMENT_SKIPPED, Option.of(fileId), Option.of(targetType),
                Option.empty(), Option.of(weight), Option.of(lastAccessTime));
    }

    public static DocviewerTskvEvent documentAccessed(String fileId, TargetType targetType, long weight, Instant lastAccessTime) {
        return new DocviewerTskvEvent(ActionEventType.DOCUMENT_ACCESSED, Option.of(fileId), Option.of(targetType),
                Option.empty(), Option.of(weight), Option.of(lastAccessTime));
    }

    public static DocviewerTskvEvent imageAccessed(String fileId, String subId) {
        return new DocviewerTskvEvent(ActionEventType.IMAGE_ACCESSED, Option.of(fileId), Option.empty(), Option.of(subId),
                Option.empty(), Option.empty());
    }

    public static DocviewerTskvEvent imageDeleted(String fileId, String subId) {
        return new DocviewerTskvEvent(ActionEventType.IMAGE_DELETED, Option.of(fileId), Option.empty(), Option.of(subId),
                Option.empty(), Option.empty());
    }

    public static DocviewerTskvEvent imageSkipped(String fileId, String subId, Instant lastAccessTime) {
        return new DocviewerTskvEvent(ActionEventType.IMAGE_SKIPPED, Option.of(fileId), Option.empty(),
                Option.of(subId), Option.empty(), Option.of(lastAccessTime));
    }

    private final ActionEventType action;

    private final Option<String> documentFileId;

    private final Option<TargetType> documentTargetType;

    private final Option<String> imageSubId;

    private final Option<Long> weight;

    private final Option<Instant> lastAccessTime;

    public ListF<Tuple2<String, String>> logData() {
        ListF<Tuple2<String, String>> result = Cf.arrayList();
        result.add(Tuple2.tuple("action", action.toString()));
        documentFileId.ifPresent(fileId -> result.add(Tuple2.tuple("file_id", fileId)));
        documentTargetType.ifPresent(targetType -> result.add(Tuple2.tuple("target_type", targetType.toString())));
        imageSubId.ifPresent(subId -> result.add(Tuple2.tuple("image_sub_id", subId)));
        weight.ifPresent(weightValue -> result.add(Tuple2.tuple("weight", String.valueOf(weightValue))));
        lastAccessTime.map(instant -> new DateTime(instant, MoscowTime.TZ).toLocalDate())
                .ifPresent(date -> result.add(Tuple2.tuple("last_access_date", lastAccessDateFormatter.print(date))));
        return result.unmodifiable();
    }

    public void log() {
        LoggerEventsRecorder.logEvent(this);
    }

}
