package ru.yandex.chemodan.app.docviewer.log;

import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.function.Function;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.CacheResult;
import ru.yandex.chemodan.app.docviewer.graphite.GraphiteEventStatisticManager;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.time.TimeUtils;

import static ru.yandex.chemodan.app.docviewer.log.DvStages.CHECK_SRC_FILE_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.CLEANUP_MULCA_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.CLEANUP_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.CONVERT_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.COPY_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.EXTRACT_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.FORWARD_TO_START_INTERNAL_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.GET_MULCA_ID;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.PROCESSING_DONE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.SCHEDULE_CONVERT;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.START_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.STORE_RESULT_STAGE;
import static ru.yandex.chemodan.app.docviewer.log.DvStages.SUBMIT_STAGE;
import static ru.yandex.chemodan.app.docviewer.utils.MimeUtils.detectExtension;

/**
 * XXX why is quote used so randomly? Need to uniform.
 * @author akirakozov
 */
public class LoggerEventsRecorder {
    private static final Logger logger = LoggerFactory.getLogger(LoggerEventsRecorder.class);
    private static final Logger tskvLogger = LoggerFactory.getLogger(DocviewerLogHelper.eventsLogTskvName);
    private static final String WARM_UP = "warm-up";
    private static final String CONVERTER = "converter";
    private static final String FILE_ID = "file_id";
    private static final String CONVERT_TYPE = "convert-type";
    private static final String MULCA_ID = "mulca-id";
    private static final String SRC_URL = "src-url";
    public static final String DST_FORMAT = "dst_format";
    private static final String TSKV_SEPARATOR = "\t";

    // akirakozov: in cocaine we should use old format, because it's used for stages graphics on dashboard
    // https://stat.yandex-team.ru/Dashboard/User/eyalunin/Docviewer?tab=100bq
    private static boolean useTskv = false;

    public static void logEvent(DocviewerTskvEvent event) {
        if (!useTskv) {
            return;
        }
        tskvLogger.info(event.logData().map(entry -> String.format("%s=%s", entry._1, entry._2))
                .mkString(TSKV_SEPARATOR));
    }

    private static String quote(String msg) {
        return "\"" + msg + "\"";
    }

    private static String describe(Object value) {
        if (value instanceof String) {
            return (String) value;
        } else if (value instanceof ActualUri) {
            return quote(((ActualUri) value).getUriString());
        } else if (value instanceof URI) {
            return quote(value.toString());
        } else if (value instanceof TargetType) {
            return ((TargetType) value).name().toLowerCase();
        } else if (value instanceof Option) {
            Option<?> opt = (Option<?>) value;
            if (opt.isPresent()) {
                return describe(opt.get());
            } else {
                return "none";
            }
        } else {
            return value.toString();
        }
    }

    private static void logStage(DvStages stage, Duration duration, String status, Tuple2List<String, Object> additionalInfo) {
        logStage(stage, Option.empty(), duration, Option.of(status), additionalInfo);
    }

    private static void logStage(DvStages stage, Exception e, Duration duration, Tuple2List<String, Object> additionalInfo) {
        logStage(stage, Option.of(e), duration, Option.empty(), additionalInfo);
    }

    private static void logStage(
            DvStages stage, Option<Exception> e, Duration duration,
            Option<String> status, Tuple2List<String, Object> additionalInfo)
    {
        String stageName = stage.getName();

        boolean success = !e.isPresent();
        String successMessage = success ? getSuccessfulStageSuccessMessage() : getFailedStageStatusSuccessMessage(e.get());
        String statusMessage = success ? status.get() : formatException(e.get());

        GraphiteEventStatisticManager.pushEventStatistic(stageName, duration, successMessage, Instant.now());
        EventsStatistic.updateGlobalStageInfo(stageName, duration, success);

        logStage(stageName, successMessage, statusMessage, duration, additionalInfo);
    }

    static <T> T logStage(DvStages stage, Tuple2List<String, Object> args, Callable<T> body,
            Function<T, Tuple2List<String, Object>> resultMapper) {
        Instant start = TimeUtils.now();
        try {
            T result = body.call();
            logStage(stage, TimeUtils.toDurationToNow(start), "ok", args.plus(resultMapper.apply(result)));
            return result;
        } catch (Exception e) {
            logStage(stage, e, TimeUtils.toDurationToNow(start), args);
            throw ExceptionUtils.translate(e);
        }
    }

    private static void logStage(String stageName, String successMessage, String statusMessage,
            Duration duration, Tuple2List<String, Object> additionalInfo)
    {
        String separator = useTskv ? TSKV_SEPARATOR : ", ";
        String specificPart = additionalInfo.map2(LoggerEventsRecorder::describe).mkString(separator, "=");
        String msgFormat = StringUtils.join(
                Arrays.asList("req-id={}", "stage={}", "duration={}", "success={}", "status={}", "{}"), separator);

        Logger log = useTskv ? tskvLogger : logger;
        log.info(msgFormat, RequestIdStack.current().getOrElse(""), stageName, TimeUtils.toSecondsString(duration),
                successMessage, quote(statusMessage), specificPart);
    }

    private static String formatException(Exception e) {
        return e.getClass().getSimpleName() + ": " + e.getMessage();
    }

    private static void logMpfsCallStage(DvStages stage, URI uri, String status, Duration duration) {
        logStage(stage, duration, status, Tuple2List.fromPairs("uri", uri));
    }

    public static void saveStoreResultFailedEvent(Exception e, Duration convertDuration) {
        logStage(STORE_RESULT_STAGE, e, convertDuration, Tuple2List.tuple2List());
    }

    public static void saveStoreResultEvent(Duration duration, FileLink resultFileLink, long resultSize) {
        logStage(STORE_RESULT_STAGE, duration, "ok",
                Tuple2List.<String, Object>fromPairs("stid", resultFileLink)
                    .plus1("result_size", Long.toString(resultSize)));
    }

    public static void saveConvertFailedEvent(TargetType convertTargetType, String converter,
            Exception e, Duration convertDuration, String contentType, boolean warmUp)
    {
        EventsStatistic.updateConverters(
                convertTargetType, converter,
                convertDuration, false, detectExtension(contentType));
        logStage(CONVERT_STAGE, e, convertDuration,
                Tuple2List.<String, Object>fromPairs(DST_FORMAT, convertTargetType)
                    .plus1(CONVERTER, converter)
                    .plus1(WARM_UP, warmUp));
    }

    public static void saveConvertEvent(TargetType convertTargetType, String converter,
            Duration convertDuration, String contentType, boolean warmUp)
    {
        EventsStatistic.updateConverters(
                convertTargetType, converter,
                convertDuration, true, detectExtension(contentType));
        logStage(CONVERT_STAGE, convertDuration, "ok",
                Tuple2List.<String, Object>fromPairs(DST_FORMAT, convertTargetType)
                        .plus1(CONVERTER, converter)
                        .plus1(WARM_UP, warmUp));
    }

    public static void saveCopyFailedEvent(ActualUri actualUri, Exception e,
            Duration duration, Option<Long> size, boolean warmUp)
    {
        logStage(COPY_STAGE, e, duration,
                Tuple2List.tuple2List(size.map(Tuple2.<String, Object>consF().bind1("size")))
                        .plus1("uri", actualUri)
                        .plus1(WARM_UP, warmUp));
    }

    public static void saveCopyEvent(URI uri, Option<String> contentType, Option<String> ext, long length,
            Duration duration, CacheResult cacheResult, boolean warmUp)
    {
        logStage(COPY_STAGE, duration, "ok",
                Tuple2List.<String, Object>fromPairs("uri", uri)
                        .plus1("ext", ext)
                        .plus1("cache", cacheResult.getValue())
                        .plus1("size", Long.toString(length))
                        .plus1("content-type", contentType.map(LoggerEventsRecorder::quote))
                        .plus1(WARM_UP, warmUp));
    }

    public static void saveExtractFailedEvent(Exception e, Duration duration, String archivePath) {
        logStage(EXTRACT_STAGE, e, duration,
                Tuple2List.fromPairs("archive-path", archivePath));
    }

    public static void saveExtractEvent(Duration duration, String archivePath, long length) {
        logStage(EXTRACT_STAGE, duration, "ok",
                Tuple2List.<String, Object>fromPairs("archive-path", archivePath)
                        .plus1("length", length));
    }

    public static void saveScheduleConvertEvent(
            String fileId, String detectedContentType, String status, Duration duration, boolean warmUp)
    {
        logStage(SCHEDULE_CONVERT, duration, status,
                Tuple2List.<String, Object>fromPairs(FILE_ID, fileId)
                        .plus1("detected-content-type", quote(detectedContentType))
                        .plus1(WARM_UP, warmUp)
        );
    }

    public static void saveScheduleConvertFailed(ActualUri actualUri, Exception e, Duration duration, boolean warmUp) {
        logStage(SCHEDULE_CONVERT, e, duration,
                Tuple2List.<String, Object>fromPairs("uri", actualUri)
                        .plus1(WARM_UP, warmUp));
    }

    public static void saveSubmitEvent(PassportUidOrZero uid,
            String reportedContentType, long length, String fileId, Duration duration)
    {
        // XXX: do we really need this info?
        logStage(SUBMIT_STAGE, duration, "",
                Tuple2List.<String, Object>fromPairs("uid", uid.toString())
                        .plus1("content-type", quote(reportedContentType))
                        .plus1("size", Long.toString(length))
                        .plus1(FILE_ID, fileId));
    }

    public static void saveStartEvent(String srcUrl, Duration duration, String status, boolean warmUp) {
        logStage(START_STAGE, duration, status,
                Tuple2List.<String, Object>fromPairs(SRC_URL, quote(UrlUtils.urlEncode(srcUrl)))
                .plus1(WARM_UP, warmUp)
        );
    }

    public static void saveStartFailedEvent(String srcUrl, Duration duration, Exception e, boolean warmUp) {
        logStage(START_STAGE, e, duration,
                Tuple2List.<String, Object>fromPairs(SRC_URL, quote(UrlUtils.urlEncode(srcUrl)))
                .plus1(WARM_UP, warmUp)
        );
    }

    private static void logCleanupSuccess(String target, Tuple2List<String, Object> additionalInfo) {
        logStage(CLEANUP_STAGE, Duration.ZERO, "ok",
                Tuple2List.<String, Object>fromPairs("target", target).plus(additionalInfo));
    }

    private static void logCleanupFail(String target, Exception e, Tuple2List<String, Object> additionalInfo) {
        logStage(CLEANUP_STAGE, e, Duration.ZERO,
                Tuple2List.<String, Object>fromPairs("target", target).plus(additionalInfo));
    }

    public static void saveCleanupUriEvent(String target, ActualUri actualUri) {
        logCleanupSuccess(target, Tuple2List.fromPairs("uri", actualUri));
    }

    public static void saveCleanupUriFailedEvent(String target, Exception e, ActualUri actualUri) {
        logCleanupFail(target, e, Tuple2List.fromPairs("uri", actualUri));
    }

    public static void saveCleanupResultEvent(String target, String fileId, Option<TargetType> convertTargetType) {
        logCleanupSuccess(target,
                Tuple2List.<String, Object>fromPairs(FILE_ID, fileId).plus1("convert-type", convertTargetType));
    }

    public static void saveCleanupResultFailedEvent(
            String target, Exception e, String fileId, Option<TargetType> convertTargetType)
    {
        logCleanupFail(target, e,
                Tuple2List.<String, Object>fromPairs(FILE_ID, fileId).plus1(CONVERT_TYPE, convertTargetType));
    }

    public static void saveCleanupFromMulcaEvent(String mulcaId) {
        logCleanupSuccess(CLEANUP_MULCA_STAGE.getName(), Tuple2List.fromPairs(MULCA_ID, mulcaId));
    }

    public static void saveCleanupFromMulcaFailedEvent(String mulcaId, Exception e) {
        logCleanupFail(CLEANUP_MULCA_STAGE.getName(), e, Tuple2List.fromPairs(MULCA_ID, mulcaId));
    }

    public static void saveCheckSourceFileEvent(URI uri, String status, Duration duration) {
        logMpfsCallStage(CHECK_SRC_FILE_STAGE, uri, status, duration);
    }

    public static void saveGetMulcaIdEvent(URI uri, String status, Duration duration) {
        logMpfsCallStage(GET_MULCA_ID, uri, status, duration);
    }

    public static void saveForwardToStartInternalEvent(String srcUrl, PassportUidOrZero uid, Duration duration, String status) {
        logStage(FORWARD_TO_START_INTERNAL_STAGE, duration, status,
                Tuple2List.<String, Object>fromPairs(SRC_URL, quote(UrlUtils.urlEncode(srcUrl)))
                        .plus1("uid", uid));
    }

    public static void saveForwardToStartInternalFailedEvent(String srcUrl, PassportUidOrZero uid, Duration duration, Exception e) {
        logStage(FORWARD_TO_START_INTERNAL_STAGE, e, duration,
                Tuple2List.<String, Object>fromPairs(SRC_URL, quote(UrlUtils.urlEncode(srcUrl)))
                        .plus1("uid-id", uid));
    }

    public static void documentProcessingDone(String metaStage, StartConversionInfo startInfo,
            ProcessingDoneInfo info)
    {
        Duration duration = new Duration(startInfo.startTime, Instant.now());
        logStage(PROCESSING_DONE, duration, "ok",
                Tuple2List.<String, Object>fromPairs(FILE_ID, info.fileId)
                        .plus1("meta-stage", metaStage)
                        .plus1(CONVERTER, info.converterName)
                        .plus1(WARM_UP, startInfo.getSource().isWarmUp())
                        .plus1("source", StringUtils.substringBefore(startInfo.getSource().getOriginalUrl(), "://"))
                        .plus1(SRC_URL, startInfo.getSource().getOriginalUrl())
                        .plus1("state", info.state)
                        .plus1(DST_FORMAT, info.targetType));
    }

    static void setUseTskv() {
        useTskv = true;
    }

    private static String getSuccessfulStageSuccessMessage() {
        return "ok";
    }

    private static String getFailedStageStatusSuccessMessage(Exception exc) {
        if (exc instanceof UserException) {
            ErrorCode errorCode = ((UserException) exc).getErrorCode();
            return ErrorCode.isNotImportantError(errorCode) ? "fail_other" : "fail";
        } else {
            return "fail";
        }
    }
}
