package ru.yandex.chemodan.app.docviewer.monica;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.commune.monica.pusher.MonicaFieldValuesExtractorRegistry;
import ru.yandex.commune.monica.pusher.MonicaPushSchemaNames;
import ru.yandex.commune.monica.pusher.StdExtractors;
import ru.yandex.commune.monica.pusher.graphite.GraphitePusherConfiguration;
import ru.yandex.misc.monica.core.blocks.AverageData;
import ru.yandex.misc.monica.core.blocks.DistributionData;
import ru.yandex.misc.monica.core.blocks.InstrumentedData;
import ru.yandex.misc.monica.core.blocks.MetricWithHealthData;
import ru.yandex.misc.monica.core.blocks.NumberData;
import ru.yandex.misc.monica.core.blocks.StatisticData;
import ru.yandex.misc.monica.core.name.FullMetricName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.core.name.MetricNamespace;
import ru.yandex.misc.monica.core.name.NameDimension;

/**
 * @author akirakozov
 */
public class DvGraphitePusherConfiguration extends GraphitePusherConfiguration {
    private static final ListF<NameDimension> IMPORTANT_DIMENSIONS = Cf.list(
            NameDimension.ENVIRONMENT, NameDimension.DC_NAME);

    private final DvGraphiteMetricsRegistry dvGraphiteMetricsRegistry;
    private final String projectVersion;
    private final String hostname;

    public DvGraphitePusherConfiguration(Duration pushPeriod, String graphitePathPrefix,
        DvGraphiteMetricsRegistry dvGraphiteMetricsRegistry, String projectVersion, String hostname)
    {
        super(pushPeriod, graphitePathPrefix, Cf.list(), createDvRegistry());
        this.dvGraphiteMetricsRegistry = dvGraphiteMetricsRegistry;
        this.projectVersion = projectVersion.replace(".", "_");
        this.hostname = hostname;
    }

    @Override
    public boolean isMetricPushEnabled(FullMetricName fullMetricName) {
        return isMetricInConfiguredList(fullMetricName.name);
    }

    public boolean isMetricInConfiguredList(MetricName name) {
        String metricNameStr = name.asList().mkString(".");

        return dvGraphiteMetricsRegistry
                .getPatterns()
                .exists(pattern -> pattern.matches(metricNameStr));
    }

    @Override
    public String getGraphitePath(FullMetricName fullMetricName, String fieldName) {
        ListF<String> namespaceValues = getImportantNamespaceValues(fullMetricName.namespace);
        MetricName name = fullMetricName.name;
        String fieldPrefix = name.asList().last().equals(fieldName) ? "" : "." + fieldName;

        return graphitePathPrefix
                + namespaceValues.mkString(".", ".", ".")
                + hostname + "."
                + projectVersion + "."
                + name.asList().mkString(".")
                + fieldPrefix;
    }

    private ListF<String> getImportantNamespaceValues(MetricNamespace namespace) {
        ListF<String> values = Cf.arrayList();
        for (NameDimension dimension : IMPORTANT_DIMENSIONS) {
            int idx = namespace.nameDimensionSet().indexOfTs(dimension);
            if (idx >= 0) {
                String value = namespace.nameDimensionValues().get(idx);
                value = namespace.nameDimensionSet().get(idx).equals(NameDimension.HOSTNAME) ?
                    value.replace('.', '_') : value;
                values.add(value);
            }
        }
        return values;
    }

    private static MonicaFieldValuesExtractorRegistry createDvRegistry() {
        MonicaFieldValuesExtractorRegistry result = MonicaFieldValuesExtractorRegistry.cons(false);

        result.register(
                MonicaPushSchemaNames.QUANTILES_AND_RATE,
                StatisticData.class,
                StdExtractors.quantilesAndRateFromStatisticExtractor()
        );

        result.register(
                MonicaPushSchemaNames.QUANTILES,
                StatisticData.class,
                StdExtractors.quantilesFromStatisticExtractor()
        );

        result.register(
                MonicaPushSchemaNames.RATE,
                StatisticData.class,
                StdExtractors.rateFromStatisticExtractor()
        );

        result.register(
                MonicaPushSchemaNames.QUANTILES_AND_RATE,
                InstrumentedData.class,
                StdExtractors.quantilesAndRateFromInstrumentedDataExtractor()
        );

        result.register(
                MonicaPushSchemaNames.QUANTILES,
                InstrumentedData.class,
                StdExtractors.quantilesFromInstrumentedDataExtractor()
        );

        result.register(
                MonicaPushSchemaNames.RATE,
                InstrumentedData.class,
                StdExtractors.rateFromInstrumentedDataExtractor()
        );

        result.register(
                MonicaPushSchemaNames.RATE,
                AverageData.class,
                StdExtractors.rateFromAverageSummarizedExtractor()
        );

        result.register(
                MonicaPushSchemaNames.SINGLE_VALUE,
                Number.class,
                StdExtractors.valueNamedAsAggregateExtractor()
        );

        result.register(
                MonicaPushSchemaNames.NUMBER_DATA,
                NumberData.class,
                StdExtractors.numberDataExtractor()
        );

        result.register(
                MonicaPushSchemaNames.SINGLE_VALUE,
                NumberData.class,
                StdExtractors.numberAllExtractor()
        );

        result.register(
                MonicaPushSchemaNames.SINGLE_VALUE,
                MetricWithHealthData.class,
                StdExtractors.fromMetricWithHealth()
        );

        result.register(
                MonicaPushSchemaNames.DISTRIBUTION_DATA_SUMMARY,
                DistributionData.class,
                StdExtractors.summaryFromDistribution()
        );

        result.registerForCustomGraphs(
                MonicaPushSchemaNames.DISTRIBUTION_DATA_SUMMARY,
                DistributionData.class,
                StdExtractors.fromDistributionForCustomGraphs()
        );

        return result;
    }

}
