package ru.yandex.chemodan.app.docviewer.monica;

import oshi.SystemInfo;
import oshi.software.os.OSProcess;
import oshi.software.os.OperatingSystem;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.ValueMap;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author akirakozov
 */
public class MemoryUsageStatistic extends DelayingWorkerServiceBeanSupport implements MonicaContainer {

    @MonicaMetric(description = "Resident set size of processes")
    @GroupByDefault
    private ValueMap<Long> rssInMb = new ValueMap<>(Long.class);
    @MonicaMetric(description = "Number of processes")
    @GroupByDefault
    private ValueMap<Integer> numberOfProcesses = new ValueMap<>(Integer.class);

    private final ListF<String> processesNames;

    public MemoryUsageStatistic(ListF<String> processesNames) {
        this.processesNames = processesNames;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "memory-usage",
                new MetricName("memory-usage"),
                "Resident set size of processes"
        );
    }

    @Override
    protected void execute() {
        ListF<OSProcess> processes = getListOfProcesses();
        addUsageByProcessName(processes, Option.empty());
        for (String name : processesNames) {
            addUsageByProcessName(processes, Option.of(name));
        }
    }

    private void addUsageByProcessName(ListF<OSProcess> processes, Option<String> processName) {
        DataSize sum = DataSize.ZERO;
        if (processName.isPresent()) {
            processes = processes.filter(p -> p.getName().contains(processName.get()));
            numberOfProcesses.set(processes.size(), processName.get());
        }
        for (OSProcess process : processes) {
            sum = sum.plus(DataSize.fromBytes(process.getResidentSetSize()));
        }
        rssInMb.set(sum.toMegaBytes(), processName.getOrElse("all"));
    }

    private static ListF<OSProcess> getListOfProcesses() {
        OperatingSystem operatingSystem = new SystemInfo().getOperatingSystem();
        return Cf.list(operatingSystem.getProcesses(0, OperatingSystem.ProcessSort.MEMORY));
    }

    public static String getCurrentStatus() {
        StringBuilder sb = new StringBuilder();
        for (OSProcess process : getListOfProcesses()) {
            sb.append(process.getName());
            sb.append(": ");
            sb.append(DataSize.fromBytes(process.getResidentSetSize()).toPrettyString());
            sb.append("\n");
        }
        return sb.toString();
    }
}
