package ru.yandex.chemodan.app.docviewer.states;

import java.util.Map;
import java.util.stream.Collectors;

import lombok.Data;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.dataSize.DataSize;

/**
 * @author ssytnik
 */
@Data
public class MaxFileSizeCheckerImpl implements MaxFileSizeChecker {

    @Value("${copier.maxFileLength}")
    private DataSize maxFileLength;

    @Value("${copier.maxArchiveLength}")
    private DataSize maxArchiveLength;

    private final DynamicProperty<ListF<String>> exts =
            new DynamicProperty<>("copier.max-file-length-extensions-with-sizes",
                    Cf.list(MimeTypes.MIME_PDF + ":2g", MimeTypes.MIME_MICROSOFT_EXCEL + ":120m",
                            MimeTypes.MIME_MICROSOFT_OOXML_EXCEL + ":120m"));

    @Autowired
    private MimeDetector mimeDetector;

    Map<String, DataSize> getSizeConfigs() {
        return buildConfig(exts.get(), getMaxFileLength());
    }

    Map<String, DataSize> buildConfig(ListF<String> strings, DataSize defaultSize) {
        return strings.stream().map(s -> s.toLowerCase().split(":"))
                .collect(Collectors.toMap(e -> e[0], e -> e.length == 1 ? defaultSize : DataSize.valueOf(e[1])));
    }

    @Override
    public DataSize getMaxFileLength(String mimeType) {
        boolean isArchive = MimeDetector.isArchive(mimeType);
        return isArchive ? getMaxArchiveLength() : getSizeLimitForMimeType(mimeType);
    }

    private DataSize getSizeLimitForMimeType(String mimeType) {
        Map<String, DataSize> sizeConfigs = getSizeConfigs();
        DataSize size = Option.ofNullable(sizeConfigs.get(mimeType)).getOrElse(getMaxFileLength());
        return getMaxFileLength().compareTo(size) > 0 ? getMaxFileLength() : size;
    }

    public Tuple2<DataSize, DataSize> setMaxLengthsForTests(Tuple2<DataSize, DataSize> maxLengths) {
        Tuple2<DataSize, DataSize> backup = Tuple2.tuple(maxFileLength, maxArchiveLength);
        this.maxFileLength = maxLengths.get1();
        this.maxArchiveLength = maxLengths.get2();
        return backup;
    }
}
