package ru.yandex.chemodan.app.docviewer.states;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.dom4j.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.adapters.mongo.MongoDbUtils;
import ru.yandex.chemodan.app.docviewer.convert.result.PageInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.misc.image.Dimension;

public class PagesInfoHelper {
    public static Option<PagesInfo> load(Object dbListObject) {
        if (!(dbListObject instanceof List))
            return Option.empty();

        @SuppressWarnings("unchecked")
        List<DBObject> list = (List<DBObject>) dbListObject;
        List<PageInfo> result = new ArrayList<>(list.size());
        for (DBObject dbObject : list) {
            final Option<Integer> index = MongoDbUtils.getInteger(dbObject, "index");
            final Option<Float> width = MongoDbUtils.getFloat(dbObject, "width");
            final Option<Float> height = MongoDbUtils.getFloat(dbObject, "height");
            result.add(new PageInfo(index, width, height, Option.empty()));
        }
        return Option.of(new PagesInfo(Cf.toList(result)));
    }

    private static Function<PageInfo, DBObject> pageInfoToDBObjectF() {
        return pageInfo -> {
            DBObject dbObject = new BasicDBObject(3);
            if (pageInfo.getIndex().isPresent())
                dbObject.put("index", pageInfo.getIndex().get());
            if (pageInfo.getWidth().isPresent())
                dbObject.put("width", pageInfo.getWidth().get());
            if (pageInfo.getHeight().isPresent())
                dbObject.put("height", pageInfo.getHeight().get());
            return dbObject;
        };
    }

    public static DBObject toDBObject(PagesInfo pagesInfo) {
        BasicDBList basicDBList = new BasicDBList();
        basicDBList.addAll(Cf.x(pagesInfo.getPageInfos()).map(pageInfoToDBObjectF()));
        return basicDBList;
    }

    public static PagesInfo toPagesInfo(PDDocument document) {
        final List<PDPage> pages = document.getDocumentCatalog().getAllPages();
        final List<PageInfo> list = new ArrayList<>(pages.size());
        for (int p = 1; p <= pages.size(); p++) {
            PageInfo pageInfo = PdfUtils.getSinglePageInfo(pages.get(p - 1), p);
            list.add(pageInfo);
        }
        return new PagesInfo(Cf.toList(list));
    }

    public static PagesInfo toSinglePageInfo(Dimension size) {
        float width = size.getWidth();
        float height = size.getHeight();
        PageInfo pageInfo = new PageInfo(1, width, height);
        return new PagesInfo(Cf.list(pageInfo));
    }

    public static void toXml(Element pages, PagesInfo pagesInfo) {
        final NumberFormat numberFormat = new DecimalFormat("#.######");
        for (PageInfo pageInfo : pagesInfo.getPageInfos()) {
            if (pageInfo == null)
                continue;

            Element page = pages.addElement("page");
            if (pageInfo.getIndex().isPresent())
                page.addAttribute("index", String.valueOf(pageInfo.getIndex().get()));
            if (pageInfo.getWidth().isPresent())
                page.addAttribute("width",
                        String.valueOf(Math.round(pageInfo.getWidth().get())));
            if (pageInfo.getHeight().isPresent())
                page.addAttribute("height",
                        String.valueOf(Math.round(pageInfo.getHeight().get())));

            if (pageInfo.getWidth().isPresent() && pageInfo.getHeight().isPresent()) {
                float width = pageInfo.getWidth().get();
                float height = pageInfo.getHeight().get();

                if (width != 0)
                    page.addAttribute("height-to-width", numberFormat.format(height / width));
            }
        }
    }
}
