package ru.yandex.chemodan.app.docviewer.states;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicReference;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.misc.concurrent.CountDownLatches;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author akirakozov
 */
public class StartManager {

    private static final Logger logger = LoggerFactory.getLogger(StartManager.class);

    @Autowired
    private StateMachine stateMachine;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private StateListenerManager stateListenerNotifier;

    public State startAndWaitUntilComplete(
            DocumentSourceInfo source, Option<String> contentType,
            TargetType targetType, Duration timeout, boolean complete)
    {
        return startAndWaitUntilComplete(
                source, contentType, targetType, timeout,
                complete, "", Cf.list(), true, Function0V.nop(), Instant.now());
    }

    public State startAndWaitUntilComplete(
            final DocumentSourceInfo source,
            final Option<String> contentType, final TargetType targetType,
            Duration timeout, final boolean complete,
            String sessionId, ListF<String> skipMimes,
            final boolean waitNotLocalConversion,
            Function0V afterStartCallback, Instant startTime)
    {
        final CountDownLatch latch = new CountDownLatch(1);
        final ActualUri uri = uriHelper.rewrite(source);
        final AtomicReference<String> fileIdRef = new AtomicReference<>();

        final Function0V fileListener = createFileListener(uri, targetType, latch);
        final Function0V uriListener = createUriListener(uri, fileIdRef, targetType, latch, fileListener, complete);

        Instant dueTime = new Instant().plus(timeout);
        stateListenerNotifier.addUriListener(uri, uriListener);

        try {
            boolean wasStarted = stateMachine.onStart(
                    source, "stage listener",
                    contentType, targetType,
                    sessionId, skipMimes, startTime, false);
            afterStartCallback.apply();

            if (wasStarted) {
                CountDownLatches.await(latch, timeout);
            } else if (waitNotLocalConversion) {
                while (dueTime.isAfterNow()) {
                    State state = stateMachine.getState(uri, targetType);
                    if (!state.isSubjectTochange() || (!complete && state.isCopyingDone())) {
                        break;
                    }
                    ThreadUtils.sleep(200);
                }
            }
        } finally {
            stateListenerNotifier.removeUriListener(uri, uriListener);
            if (fileIdRef.get() != null) {
                stateListenerNotifier.removeFileListener(fileIdRef.get(), uriListener);
            }
        }

        return stateMachine.getState(uri, targetType);
    }

    private Function0V createUriListener(
            ActualUri uri, AtomicReference<String> fileIdRef,
            TargetType targetType, CountDownLatch latch,
            Function0V fileListener, boolean waitUntilComplete)
    {
        return () -> {
                State state = stateMachine.getState(uri, targetType);
                logger.debug("Uri listener, state: {}", state);
                if (!state.isSubjectTochange()) {
                    latch.countDown();
                } else if (state == State.CONVERTING) {
                    if (waitUntilComplete) {
                        String fileId = stateMachine.getFileId(uri)
                                .getOrThrow("State is CONVERTING, but file-id is not found");
                        fileIdRef.set(fileId);
                        stateListenerNotifier.addFileListener(fileId, fileListener);
                    } else {
                        latch.countDown();
                    }
                }
            };
    }

    private Function0V createFileListener(ActualUri uri, TargetType targetType, CountDownLatch latch) {
        return () -> {
                State state = stateMachine.getState(uri, targetType);
                logger.debug("File listener, state: {}", state);
                if (!state.isSubjectTochange()) {
                    latch.countDown();
                }
            };
    }
}
