package ru.yandex.chemodan.app.docviewer.states;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import lombok.SneakyThrows;
import lombok.val;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.docviewer.convert.ConvertArgs;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.ExConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.CopiedFileInfo;
import ru.yandex.chemodan.app.docviewer.copy.Copier;
import ru.yandex.chemodan.app.docviewer.copy.CopyInfo;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.StoredUriManager;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.results.ConvertErrorArgs;
import ru.yandex.chemodan.app.docviewer.dao.results.ConvertSuccessArgs;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionDao;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey.SessionConvertPassword;
import ru.yandex.chemodan.app.docviewer.dao.sessions.SessionKey.SessionCopyPassword;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.log.DocviewerTskvEvent;
import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.log.ProcessingDoneInfo;
import ru.yandex.chemodan.app.docviewer.log.StartConversionInfo;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.utils.Digester;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.util.HostBasedSwitch;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.inside.passport.tvm2.UserTicketHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.version.Version;

import static ru.yandex.chemodan.app.docviewer.convert.TargetType.PDF;

public class StateMachineImpl implements StateMachine {

    private static final Logger logger = LoggerFactory.getLogger(StateMachineImpl.class);

    public final DynamicProperty<Long> resultsYandexWeight =
            new DynamicProperty<>("docviewer.results.yandex.weight", 365 * 2 * 1000L);

    public final DynamicProperty<Boolean> useRemoteIdForHardlink =
            new DynamicProperty<>("docviewer.results.use.remote-id.hardlink", false);

    public final DynamicProperty<Long> preferHtmlForPdfBiggerThen =
            new DynamicProperty<>("docviewer.results.prefer.html_for_pdf.kb", 100 * 1024L);

    private final DynamicProperty<Long> needToStartCopyDeadline = new DynamicProperty<>("docviewer.need-to-start-copy-deadline.ms", 5000L);
    private final HostBasedSwitch forceCopyExternal = new HostBasedSwitch("docviewer.force-copy-external.hosts");
    private final DynamicProperty<Integer> maxPageCountInTheLog =
            new DynamicProperty<>("docviewer.results.max-page-count-in-the-log", 1000);

    @Autowired
    private ConvertManager convertManager;
    @Autowired
    private Copier copier;
    @Autowired
    private MimeDetector mimeDetector;
    @Autowired
    private UriRightsDao uriRightsDao;
    @Autowired
    private StateListenerNotifier stateListenerManager;
    @Autowired
    private StoredResultDao storedResultDao;
    @Autowired
    private StoredUriDao storedUriDao;
    @Autowired
    private SessionDao sessionDao;
    @Autowired
    private StoredUriManager storedUriManager;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private Version docviewerWebVersion;
    @Autowired
    private UserErrorReultsChecker userErrorReulstChecker;
    @Autowired
    private YandexStaffUserRegistry yandexStaffUserRegistry;
    @Autowired
    private Scheduler copierScheduler;
    @Autowired
    private MaxFileSizeChecker maxFileSizeChecker;

    private static class ConvertionScheduleResult {
        final boolean isFileUsed;
        final String statusMessage;
        final String digest;
        final String detectedContentType;

        ConvertionScheduleResult(boolean isFileUsed, String statusMessage, String digest, String detectedContentType) {
            this.isFileUsed = isFileUsed;
            this.statusMessage = statusMessage;
            this.digest = digest;
            this.detectedContentType = detectedContentType;
        }
    }

    private void addConvertState(final String fileId, Element result, String sessionId) {
        for (TargetType convertTargetType : convertManager.getEnabledTargetTypes()) {

            Element convertStatus = DocumentHelper.createElement("convert-state");
            convertStatus.addAttribute("target", convertTargetType.name());
            result.add(convertStatus);

            State convertState = State.NOT_STARTED;
            try {
                if (convertManager.isScheduled(fileId, convertTargetType)) {
                    convertState = State.CONVERTING;
                }

                Option<StoredResult> storedResultO = storedResultDao.find(fileId, convertTargetType);

                if (!storedResultO.isPresent()) {
                    continue;
                }

                StoredResult storedResult = storedResultO.get();

                if (storedResult.getErrorCode().isPresent()) {
                    convertState = State.CONVERTING_ERROR;
                    addConvertError(convertStatus,
                            storedResult.getError().getOrElse(""),
                            storedResult.getErrorCode().getOrElse(ErrorCode.UNKNOWN_CONVERT_ERROR));
                    continue;
                }

                if (!validateConvertPassword(storedResult, sessionId)) {
                    convertState = State.CONVERTING_ERROR; // access convert error -- separate error?
                    addConvertError(convertStatus, "Convert access error", ErrorCode.FILE_IS_PASSWORD_PROTECTED);
                    continue;
                }

                convertState = State.AVAILABLE;

                // XXX 2012-12-14 temporary migration code: none -> ""
                // is supported for backward-compatibility reasons only;
                // for now, makeup should take care of ARCHIVE_LISTING value
                // TODO replace with storedResult.getConvertResultType().get()
                convertStatus.addElement("result-type").addText(
                        storedResult.getConvertResultType()
                        .map(Enum::name).getOrElse(""));

                XmlUtils2.appendProperties(convertStatus, storedResult.getDocumentProperties());

                storedResult.getPages().ifPresent(pages -> {
                    Element pagesElement = convertStatus.addElement("pages");
                    pagesElement.addAttribute("count", String.valueOf(pages));
                    storedResult.getPagesInfo().ifPresent(info -> PagesInfoHelper.toXml(pagesElement, info));
                });
            } finally {
                convertStatus.addElement("state").addText(String.valueOf(convertState));
            }
        }
    }

    private void addConvertError(Element convertStatus, String text, ErrorCode code) {
        convertStatus.addElement("convert-error").addText(text);
        convertStatus.addElement("convert-error-code").addText(code.name());
    }

    private boolean validateConvertPassword(StoredResult storedResult, String sessionId) {
        logger.debug("Checking if session with id '{}' has access to file-id '{}': ", sessionId,
                storedResult.getFileId());

        for (String hash : storedResult.getPasswordHash()) {
            SessionConvertPassword key = new SessionConvertPassword(storedResult.getFileId());
            Option<String> password = sessionDao.findValidValue(sessionId, key);

            if (!password.isPresent()) {
                logger.debug("Missing convert password");
                return false;
            }
            if (!hash.equals(SessionKey.toHashValue(password.get()))) {
                logger.debug("Convert password mismatch");
                return false;
            }
        }

        logger.debug("Access granted, {} convert password(s) verified", storedResult.getPasswordHash().size());

        return true;
    }

    @Override
    public Option<String> getFileId(ActualUri uri) {
        return storedUriDao.find(uri).flatMapO(StoredUri::getFileId);
    }

    @Override
    public Element getInfo(ActualUri uri, String sessionId) {
        Element result = DocumentHelper.createElement("state");

        State copyState = State.NOT_STARTED;
        final Option<String> fileId;
        try {
            final Option<StoredUri> storedUriO = storedUriDao.find(uri);

            if (!storedUriO.isPresent()) {
                copyState = State.NOT_STARTED;
                return result;
            }

            final StoredUri storedUri = storedUriO.get();

            if (storedUri.getError().isPresent()) {
                copyState = State.COPY_ERROR;
                addCopyError(result, storedUri);
                return result;
            }

            fileId = storedUri.getFileId();
            if (!fileId.isPresent()) {
                copyState = State.COPYING;
                return result;
            }

            Option<String> accessErrorArchivePath = validateCopyPasswords(storedUri, sessionId);
            if (accessErrorArchivePath.isPresent()) {
                copyState = State.COPY_ERROR; // access copy error -- separate error?
                addCopyError(result, "Copy access error", ErrorCode.FILE_IS_PASSWORD_PROTECTED,
                        accessErrorArchivePath, Option.empty());
                return result;
            }

            result.addElement("file-id").addText(fileId.get());
            copyState = State.COPIED;

            final String detectedContentType = storedUri.getContentType();
            if (StringUtils.isNotEmpty(detectedContentType)) {
                result.addElement("detected-content-type").addText(detectedContentType);
                result.addElement("detected-content-family").addText(
                        mimeDetector.getContentTypeFamily(detectedContentType));
            }

            storedUri.getSerpLastAccess()
                    .ifPresent(v -> result.addElement("serp-last-access").addText(String.valueOf(v.getMillis())));
            storedUri.getContentSize().ifPresent(v -> result.addElement("size").addText(v.toString()));
            if (isPreferHtmlWithImages(storedUri)) {
                result.addElement("prefer_html_with_images").addText(String.valueOf(true));
            }
        } finally {
            result.addElement("state").addText(String.valueOf(copyState));
        }

        fileId.ifPresent(v -> addConvertState(v, result, sessionId));

        return result;
    }

    private boolean isPreferHtmlWithImages(StoredUri storedUri) {
        if (storedUri.getConvertTargets().containsKey(PDF)
                && storedUri.getContentSize().isMatch(cs -> cs.gt(DataSize.fromKiloBytes(preferHtmlForPdfBiggerThen.get()))))
        {
            return true;
        } else if (StringUtils.isNotEmpty(storedUri.getContentType())
                && Cf.list("pdf", "djvu").containsTs(mimeDetector.getContentTypeFamily(storedUri.getContentType()))
                && storedUri.getContentSize().isMatch(cs -> cs.gt(DataSize.fromKiloBytes(preferHtmlForPdfBiggerThen.get()))))
        {
            return true;
        }
        return false;
    }

    private void addCopyError(Element result, StoredUri storedUri) {
        Option<DataSize> maxSize = storedUri.getMaxContentSize();
        if (!maxSize.isPresent() && (
                storedUri.getErrorCode().isSome(ErrorCode.FILE_TOO_BIG) ||
                storedUri.getErrorCode().isSome(ErrorCode.ARCHIVE_TOO_BIG)))
        {
            maxSize = Option.of(maxFileSizeChecker.getMaxFileLength(storedUri.getContentType()));
        }

        addCopyError(result, storedUri.getError().getOrElse(""), storedUri.getErrorCode().getOrElse(ErrorCode.UNKNOWN_COPY_ERROR),
                storedUri.getErrorArchivePath(), maxSize.map(DataSize::toBytes).map(size -> Long.toString(size)));
    }

    private void addCopyError(Element result, String text, ErrorCode code, Option<String> errorArchivePath, Option<String> maxSize) {
        result.addElement("copy-error").addText(text);
        result.addElement("copy-error-code").addText(code.name());
        errorArchivePath.ifPresent(v -> result.addElement("copy-error-archive-path").addText(v));
        maxSize.ifPresent(s -> result.addElement("limit-length").addText(s));
    }

    private Option<String> validateCopyPasswords(StoredUri storedUri, String sessionId) {
        logger.debug("Checking if session with id '{}' has access to uri '{}': ", sessionId, storedUri.getUri());

        for (Tuple2<String, String> entry : storedUri.getPasswords()) {
            String path = entry.get1();
            String hash = entry.get2();

            SessionCopyPassword key = new SessionCopyPassword(storedUri.getUri().withArchivePath(path));
            Option<String> password = sessionDao.findValidValue(sessionId, key);

            if (!password.isPresent()) {
                logger.debug("Missing copy password for archive path: '{}'", path);
                return Option.of(path);
            }
            if (!hash.equals(SessionKey.toHashValue(password.get()))) {
                logger.debug("Copy password mismatch for archive path: '{}'", path);
                return Option.of(path);
            }
        }

        logger.debug("Access granted, {} copy password(s) verified", storedUri.getPasswords().size());

        return Option.empty();
    }


    @Override
    public Element getInfo(PassportUidOrZero uid, String fileId, String sessionId) {
        Element result = DocumentHelper.createElement("state");
        result.addElement("file-id").addText(fileId);

        Option<StoredUri> storedUriO = storedUriManager.findByFileIdAndUidO(fileId, uid);

        if (storedUriO.isPresent()) {
            final StoredUri storedUri = storedUriO.get();

            final String detectedContentType = storedUri.getContentType();
            if (StringUtils.isNotEmpty(detectedContentType)) {
                result.addElement("detected-content-type").addText(detectedContentType);
                result.addElement("detected-content-family").addText(
                        mimeDetector.getContentTypeFamily(detectedContentType));
            }

            result.addElement("state").addText(String.valueOf(State.COPIED));

        } else {
            result.addElement("state").addText(String.valueOf(State.NOT_FOUND));
            return result;
        }

        addConvertState(fileId, result, sessionId);
        return result;
    }

    @Override
    public State getState(ActualUri uri, TargetType convertTargetType) {
        return getState(storedUriDao.find(uri), convertTargetType);
    }

    private State getState(Option<StoredUri> storedUriO, TargetType convertTargetType) {
        return storedUriO.map(storedUri -> {
            if (storedUri.getError().isPresent()) {
                return State.COPY_ERROR;
            }

            return storedUri.getFileId().map(fileId -> {
                if (!storedUri.getConvertTargets().containsKey(convertTargetType)) {
                    return State.NOT_STARTED;
                }

                return storedResultDao.find(fileId, convertTargetType).map(storedResult -> {
                    if (storedResult.getError().isPresent()) {
                        return State.CONVERTING_ERROR;
                    }

                    return State.AVAILABLE;
                }).getOrElse(() -> {
                    if (convertManager.isScheduled(fileId, convertTargetType)) {
                        return State.CONVERTING;
                    }

                    return State.COPIED;
                });
            }).getOrElse(() -> {
                if (copierScheduler.isScheduled(storedUri.getUri().getUriString())) {
                    return State.COPYING;
                } else {
                    return State.NOT_STARTED;
                }
            });
        }).getOrElse(State.NOT_STARTED);
    }

    @Override
    public State getState(PassportUidOrZero uid, String fileId, TargetType targetType) {
        return storedUriManager.findByFileIdAndUidO(fileId, uid).map(
                storedUri -> storedResultDao.find(fileId, targetType)
                    .map(storedResult -> storedResult.getError().isPresent() ? State.CONVERTING_ERROR : State.AVAILABLE)
                    .getOrElse(() -> convertManager.isScheduled(fileId, targetType) ? State.CONVERTING : State.NOT_STARTED)
        ).getOrElse(State.NOT_FOUND);
    }

    @Override
    public void onConvertBegin(String fileId, TargetType convertTargetType) {
        logger.debug("onConvertBegin('{}', {})", fileId, convertTargetType);
    }

    @Override
    public void onConvertDone(ConvertArgs convertArgs, FileLink resultFileLink, ExConvertResultInfo result, Option<String> password) {
        int maxPageCount = maxPageCountInTheLog.get();

        if (result.getResultInfo().getPagesInfo().filter(p -> p.getCount() > maxPageCount).isPresent()) {
            PagesInfo trunkedPageInfo = new PagesInfo(result.getResultInfo().getPagesInfo().get().getPageInfos().take(maxPageCount));
            ExConvertResultInfo trunkedResult = result.toBuilder()
                    .resultInfo(result.getResultInfo()
                            .toBuilder()
                            .pagesInfo(Option.of(trunkedPageInfo))
                            .build())
                    .build();
            logger.debug("truncated from {} to {} pages: onConvertDone('{}', {}, {})",
                    result.getResultInfo().getPagesInfo().get().getCount(), maxPageCount,
                    convertArgs.getFileId(), convertArgs.getTargetType(), trunkedResult);
        } else {
            logger.debug("onConvertDone('{}', {}, {})",
                    convertArgs.getFileId(), convertArgs.getTargetType(), result);
        }

        ConvertResultInfo convertResultInfo = result.getResultInfo();
        storedResultDao.saveOrUpdateResult(ConvertSuccessArgs.builder().fileId(convertArgs.getFileId())
                .contentType(Option.ofNullable(convertArgs.getContentType()))
                .remoteFileId(convertArgs.getStartInfo().getSource().getRemoteFileId())
                .targetType(convertArgs.getTargetType()).type(convertResultInfo.getType())
                .resultFileLink(resultFileLink.toString()).length(result.getResultFile().length())
                .pages(convertResultInfo.getPages()).rawPassword(password)
                .pagesInfo(convertResultInfo.getPagesInfo())
                .weight(calcWeight(result.getConvertTime(), convertArgs.getStartInfo().isYandex, convertArgs.getStartInfo().getSource().isWarmUp()))
                .properties(convertResultInfo.getProperties()).restoreUri(convertArgs.getRestoreUri()).build());

        stateListenerManager.onStateChange(convertArgs.getFileId());
    }

    private long calcWeight(Duration convertTime, boolean isYandex, boolean isWarmUp) {
        return isYandex ? resultsYandexWeight.get() : isWarmUp ? convertTime.getMillis() / 100 : convertTime.getMillis();
    }

    @Override
    public void onConvertError(ConvertArgs args, Exception exc) {
        logger.warn("onConvertError('" + args.getFileId() + "'): " + exc, exc);

        val errorCode = Option.of(exc).filterByType(UserException.class).map(UserException::getErrorCode).getOrElse(ErrorCode.UNKNOWN_CONVERT_ERROR);
        int newFailedAttemptsCount = args.getFailedAttemptsCount() + (errorCode == ErrorCode.FILE_IS_PASSWORD_PROTECTED ? 0 : 1);
        storedResultDao.saveOrUpdateResult(ConvertErrorArgs.builder().
                fileId(args.getFileId()).targetType(args.getTargetType()).errorCode(errorCode)
                .error(ExceptionUtils.getStackTrace(exc)).failedAttemptsCount(newFailedAttemptsCount)
                .packageVersion(docviewerWebVersion.getProjectVersion()).build());

        stateListenerManager.onStateChange(args.getFileId());
    }

    @Override
    public void onCopyBegin(ActualUri uri) {
        logger.debug("onCopyBegin('{}')", uri);
    }

    @Autowired
    private Digester digester;

    @Override
    public boolean onCopyDone(CopiedFileInfo info, String sessionId)
    {
        logger.debug("onCopyDone('{}', '{}', '{}')", info.uri,
                info.reportedContentType, info.localCopy.getFile());

        final Instant startTime = TimeUtils.now();
        try {

            ConvertionScheduleResult res = scheduleConversion(info, sessionId);

            LoggerEventsRecorder.saveScheduleConvertEvent(
                    res.digest, res.detectedContentType, res.statusMessage,
                    TimeUtils.toDurationToNow(startTime), info.startInfo.getSource().isWarmUp());
            return res.isFileUsed;
        } catch (Exception e) {
            LoggerEventsRecorder.saveScheduleConvertFailed(
                    info.uri, e, TimeUtils.toDurationToNow(startTime), info.startInfo.getSource().isWarmUp());
            throw ExceptionUtils.translate(e);
        }
    }

    private ConvertionScheduleResult scheduleConversion(CopiedFileInfo info, String sessionId) {
        Option<StoredUri> storedUriO = storedUriDao.find(info.uri);
        if (!storedUriO.isPresent() || storedUriO.get().getConvertTargets().isEmpty()) {
            String message = !storedUriO.isPresent() ?
                             "Doing nothing because URI info is not present in DB" :
                             "Doing nothing because no target convert types were specified for URI";
            return new ConvertionScheduleResult(false, message, info.fileId, "");
        } else {
            transferUriRights(info.uri, info.fileId);

            String detectedContentType = detectContentType(info);

            Map<TargetType, Float> convertTargets = storedUriO.get().getConvertTargets();

            ListF<TargetType> scheduledTargets = scheduleIfNecessary(info, convertTargets, detectedContentType, sessionId);

            DataSize size = DataSize.fromBytes(info.localCopy.getFile().length());
            storedUriDao.updateUri(info.uri, Option.of(detectedContentType), info.fileId, size, info.serpLastAccess);

            stateListenerManager.onStateChange(info.uri);
            stateListenerManager.onStateChange(info.fileId);

            ListF<TargetType> doneAlready = Cf.toList(convertTargets.keySet()).filter(scheduledTargets.containsF().notF());
            ListF<String> statusMessages = Cf.arrayList();
            if (!scheduledTargets.isEmpty()) {
                statusMessages.add("Convert scheduled for " + scheduledTargets + ".");
            }
            if (!doneAlready.isEmpty()) {
                for (TargetType targetType : doneAlready) {
                    LoggerEventsRecorder.documentProcessingDone(
                            "afterCopy", info.startInfo,
                            getProcessingInfo(info.fileId, detectedContentType, info.uri, targetType));
                }
                statusMessages.add("Types " + doneAlready + " done already or not needed to convert.");
            }

            return new ConvertionScheduleResult(scheduledTargets.isNotEmpty(),
                    StringUtils.join(statusMessages, " "), info.fileId, detectedContentType);
        }
    }

    // Here, just in case, we update all records in both tables for all uids.
    // In fact, in onStartInternal we need to do it only for current uid.
    // Though the same applies to calculateDigestAndScheduleConversion now,
    // but in future it may change to current behavior (if uris.file-id is deleted).
    private void transferUriRights(ActualUri uri, String fileId) {
        logger.debug("Transferring rights from '{}' to file '{}'...", uri, fileId);

        uriRightsDao.updateUriRights(uri, fileId);
    }

    private ListF<TargetType> scheduleIfNecessary(CopiedFileInfo info, Map<TargetType, Float> convertTargets,
            String detectedContentType, String sessionId)
    {
        ListF<TargetType> scheduledTargets = Cf.arrayList();
        TargetType convertTargetType = info.getStartInfo().getConvertTargetType();
        final Float priority = convertTargets.get(convertTargetType);
        int failedAttemptsCount = 0;
        Instant lastTry = Instant.now();
        Option<ErrorCode> errorCode = Option.empty();

        if (convertTargetType.isHtmlWithImages()
                && info.getStartInfo().skipableContentTypes.map(MimeDetector.normalizeF()).containsTs(detectedContentType))
        {
            // Some documents are showed in browser viewer in iframe,
            // docviewer shouldn't convert them (DOCVIEWER-1582)
            return scheduledTargets;
        }

        Option<StoredResult> storedResultO = storedResultDao.find(info.getFileId(), convertTargetType);
        if (storedResultO.isPresent()) {
            StoredResult storedResult = storedResultO.get();
            logger.debug("File with hash {} already processed to {} (error was: '{}').",
                    info.getFileId(), convertTargetType, storedResult.getCompactError());
            DocviewerTskvEvent.documentAccessed(info.getFileId(), convertTargetType, storedResult.getWeight(), storedResult.getLastAccess()).log();
            setRestoreUriIfMissing(storedResult, info.getRestoreUri());

            errorCode = storedResult.getErrorCode();
            if (errorCode.isPresent()) {
                if (userErrorReulstChecker.notAllowed(errorCode, storedResult.getLastAccess())) {
                    return scheduledTargets;
                }
                storedResultDao.delete(info.getFileId(), convertTargetType);
                if (wasConvertedWithSamePackage(storedResult)) {
                    failedAttemptsCount = storedResult.getFailedAttemptsCount().getOrElse(0);
                    lastTry = storedResult.getLastAccess();
                }
            } else {
                executeNonCriticalAction("update last access time",
                        () -> storedResultDao.updateLastAccessTime(info.getFileId(), convertTargetType));
                if (yandexStaffUserRegistry.isStaffUser(info.getStartInfo().getSource().getUid())) {
                    logger.info("Yandex user, changing cache weight: {}, {}", info.getStartInfo().getSource().getUid(), info.getFileId());
                    executeNonCriticalAction("update weight", () -> storedResultDao.updateWeight(
                            info.getFileId(), convertTargetType, resultsYandexWeight.get()));
                }
                return scheduledTargets;
            }
        }

        convertManager.scheduleConvert(ConvertArgs.builder().fileId(info.getFileId()).targetType(convertTargetType)
                .contentType(detectedContentType).localCopy(info.getLocalCopy()).priority(priority)
                .sessionId(sessionId).failedAttemptsCount(failedAttemptsCount).lastTry(lastTry)
                .startInfo(info.getStartInfo()).restoreUri(info.getRestoreUri()).errorCode(errorCode)
                .build());

        scheduledTargets.add(convertTargetType);

        return scheduledTargets;
    }

    String detectContentType(CopiedFileInfo info) {
        String detectedContentType;
        Set<String> possibleContentTypes = new LinkedHashSet<>(4);
        possibleContentTypes.addAll(info.reportedContentType.map(Cf.String.toLowerCaseF()));
        possibleContentTypes.addAll(info.contentDispositionFilename.map(mimeDetector
                .getMimeTypeByFilenameF()));
        if (info.useUriForMimeType) {
            possibleContentTypes.add(mimeDetector.getMimeTypeByUri(info.uri.getUri()));
        }
        detectedContentType = mimeDetector.getMimeType(info.localCopy.getFile(), possibleContentTypes);
        return detectedContentType;
    }

    @Override
    public void onCopyError(ActualUri uri, Exception exc) {
        logger.warn("onCopyError(" + uri + "): " + exc, exc);

        val code = Option.of(exc).filterByType(UserException.class).map(UserException::getErrorCode).getOrElse(ErrorCode.UNKNOWN_COPY_ERROR);
        storedUriDao.updateUri(uri, code,
                ExceptionUtils.getStackTrace(exc), getErrorArchivePath(exc),
                getContentSize(exc), getMaxContentSize(exc));
        stateListenerManager.onStateChange(uri);
    }

    private Option<Long> getContentSize(Exception exc) {
        return exc instanceof FileTooBigUserException ?
               ((FileTooBigUserException) exc).getActualLength() : Option.empty();
    }

    private Option<Long> getMaxContentSize(Exception exc) {
        return exc instanceof FileTooBigUserException ?
               Option.of(((FileTooBigUserException) exc).getMaxLength()) : Option.empty();
    }

    private Option<String> getErrorArchivePath(Exception exc) {
        return exc instanceof PasswordProtectedException ?
                ((PasswordProtectedException) exc).getErrorArchivePath() : Option.empty();
    }

    @Override
    public boolean onStart(DocumentSourceInfo source, String taskSource,
            Option<String> contentType, TargetType convertTargetType,
            String sessionId, ListF<String> skipableContentTypes, Instant startTime, boolean forwarded)
    {
        logger.debug("onStart({}, '{}', '{}', {})",
                source.getUid(), source.getOriginalUrl(), source.getArchivePath().getOrElse(""), convertTargetType);
        try {
            return onStartInternal(source, contentType, convertTargetType,
                    startTime, sessionId, skipableContentTypes, forwarded);
        } catch (Exception e) {
            LoggerEventsRecorder.saveStartFailedEvent(
                    source.getOriginalUrl(), TimeUtils.toDurationToNow(startTime), e, source.isWarmUp());
            throw ExceptionUtils.translate(e);
        }
    }

    private boolean onStartInternal(DocumentSourceInfo source,
            Option<String> contentType, TargetType convertTargetType, Instant startTime,
            String sessionId, ListF<String> skipableContentTypes, boolean forwarded)
    {
        boolean isExternalUrl = uriHelper.isExternalUrl(source.getOriginalUrl());
        ActualUri actualUri = uriHelper.rewrite(source);
        uriRightsDao.saveOrUpdateUriRight(actualUri, source.getUid());

        Option<StoredUri> storedUriO = storedUriDao.find(actualUri);
        StartConversionInfo startInfo = StartConversionInfo.builder()
                .startTime(startTime)
                .source(source)
                .convertTargetType(convertTargetType)
                .storedUriO(storedUriO)
                .skipableContentTypes(skipableContentTypes)
                .isExternalUrl(isExternalUrl)
                .actualUri(actualUri)
                .isYandex(yandexStaffUserRegistry.isInitialized() &&
                        yandexStaffUserRegistry.isStaffUser(source.getUid())).build();

        if (forwarded || needToStartCopyWithDeadline(startInfo)) {
            logger.warn("Started copying {} ", actualUri);
            storedUriO.ifPresent(uri -> storedUriDao.updateUriClean(actualUri));
            storedUriDao.saveOrUpdateUri(actualUri, contentType, convertTargetType, 1f);

            copier.scheduleCopy(CopyInfo.builder()
                    .uri(actualUri)
                    .isExternalUri(isExternalUrl)
                    .sessionId(sessionId)
                    .contentType(contentType)
                    .source(source)
                    .startInfo(startInfo)
                    .forwarded(forwarded)
                    .tvmUserTicket(UserTicketHolder.get())
                    .build());

            stateListenerManager.onStateChange(actualUri);

            LoggerEventsRecorder.saveStartEvent(source.getOriginalUrl(),
                    TimeUtils.toDurationToNow(startTime), "ok", source.isWarmUp());
            return true;
        } else {
            LoggerEventsRecorder.saveStartEvent(source.getOriginalUrl(), TimeUtils.toDurationToNow(startTime),
                    "don't start new conversion, because it is treated needless", source.isWarmUp());

            Option<StoredUri> storedUriOOrFound = storedUriO.orElse(() -> storedUriDao.find(actualUri));

            executeNonCriticalAction("find storedResult by storedUri", () -> {
                storedUriOOrFound.filterMap(StoredUri::getFileId)
                        .filterMap(fileId -> storedResultDao.find(fileId, convertTargetType))
                        .forEach(storedResult -> setRestoreUriIfMissing(
                                storedResult, StoredResult.createRestoreUri(actualUri, isExternalUrl)));
            });

            // this actually wrong for COPYING and CONVERTING case as file-ids may differ, but no one cares
            return storedUriOOrFound.map(storedUri -> {
                storedUri.getFileId().ifPresent(fileId -> transferUriRights(actualUri, fileId));
                LoggerEventsRecorder.documentProcessingDone("onStart", startInfo, getProcessingInfo(storedUri, convertTargetType));
                return false;
            }).getOrElse(() -> {
                logger.warn("Stored uri should exists");
                return false;
            });
        }
    }

    private ProcessingDoneInfo getProcessingInfo(StoredUri storedUri, TargetType targetType) {
        return getProcessingInfo(storedUri.getFileId().getOrElse("-"),
                storedUri.getContentType(), storedUri.getUri(), targetType);
    }

    private ProcessingDoneInfo getProcessingInfo(String fileId, String contentType,
        ActualUri actualUri, TargetType targetType)
    {
        String converterName = convertManager.getConvertersSafelyFor(contentType).firstO()
                .map(converter -> convertManager.getConverterName(converter)).getOrElse("-");
        State state = getState(actualUri, targetType);
        return new ProcessingDoneInfo(state, targetType, fileId, converterName);
    }

    @SneakyThrows
    private boolean needToStartCopyWithDeadline(StartConversionInfo startInfo) {
        try {
            return needToStartCopy(startInfo).get(needToStartCopyDeadline.get(), TimeUnit.MILLISECONDS);
        } catch (TimeoutException e) {
            logger.warn("needToStartCopy() timed out");
            return true;
        } catch (ExecutionException e) {
            throw e.getCause();
        }
    }

    private CompletableFuture<Boolean> needToStartCopy(StartConversionInfo startInfo) {
        return startInfo.storedUriO.map(storedUri -> {
            ActualUri actualUri = storedUri.getUri();
            State state = getState(startInfo.storedUriO, startInfo.convertTargetType);
            switch (state) {
                case COPYING:
                    return CompletableFuture.completedFuture(false);
                case CONVERTING:
                    return CompletableFuture.completedFuture(false);
                case COPIED:
                    return needStartCopyOnCopiedState(storedUri, startInfo.skipableContentTypes, startInfo.isExternalUrl);
                case CONVERTING_ERROR:
                    return needRetryAfterConvertError(startInfo.convertTargetType, storedUri, startInfo.isExternalUrl);
                case AVAILABLE:
                    if (forceCopyExternal.get() && startInfo.isExternalUrl) {
                        return CompletableFuture.completedFuture(true);
                    } else {
                        return copier.needToDownloadFile(actualUri, storedUri.getTimestamp(), startInfo.isExternalUrl);
                    }
                default:
                    return checkByRemoteIdWasNotConverted(actualUri, startInfo);
            }
        }).getOrElse(() -> checkByRemoteIdWasNotConverted(startInfo.getActualUri(), startInfo));
    }

    private CompletableFuture<Boolean> checkByRemoteIdWasNotConverted(ActualUri actualUri, StartConversionInfo startInfo) {
        if (!useRemoteIdForHardlink.get() || startInfo.getSource().getArchivePath().isPresent()) {
            return CompletableFuture.completedFuture(true);
        }
        return startInfo.getSource().getRemoteFileId()
                .map(id -> storedResultDao.findByRemoteId(id)).getOrElse(Cf.list())
                .find(r -> startInfo.convertTargetType == r.getConvertTargetType())
                .filter(k -> !k.getError().isPresent())
                .filter(k -> k.getContentType().isPresent())
                .map(sr -> {
                    storedUriDao.saveOrUpdateUri(actualUri, sr.getContentType(), sr.getConvertTargetType(), 1f);
                    storedUriDao.updateUri(actualUri, sr.getContentType(), sr.getFileId(), DataSize.ZERO, Option.empty());
                    logger.info("Copied resource for {}", startInfo);
                    return sr.getLastAccess();
                })
                .map(t -> copier.needToDownloadFile(actualUri, t, startInfo.isExternalUrl))
                .getOrElse(() -> CompletableFuture.completedFuture(true));
    }

    private CompletableFuture<Boolean> needStartCopyOnCopiedState(StoredUri storedUri, ListF<String> skipableContentTypes,
        boolean isExternalUri)
    {
        ListF<String> normalizedContentTypes = skipableContentTypes.map(MimeDetector.normalizeF());
        if (normalizedContentTypes.containsTs(storedUri.getContentType())) {
            return copier.needToDownloadFile(storedUri.getUri(), storedUri.getTimestamp(), isExternalUri);
        } else {
            return CompletableFuture.completedFuture(true);
        }
    }

    private CompletableFuture<Boolean> needRetryAfterConvertError(TargetType convertTargetType, StoredUri storedUri,
        boolean isExternalUri)
    {
        String fileId = storedUri.getFileId().getOrThrow("File internal id not found");
        StoredResult resultO = storedResultDao.find(fileId, convertTargetType).getOrThrow("Result not found");
        if (!wasConvertedWithSamePackage(resultO) || !ErrorCode.isPermanentConvertError(resultO.getErrorCode().getOrThrow("Error code not found"))) {
            return CompletableFuture.completedFuture(true);
        } else {
            return copier.needToDownloadFile(storedUri.getUri(), storedUri.getTimestamp(), isExternalUri);
        }
    }

    private boolean wasConvertedWithSamePackage(StoredResult resultO) {
        return resultO.getPackageVersion().isSome(docviewerWebVersion.getProjectVersion());
    }

    private void executeNonCriticalAction(String actionName, Runnable action) {
        try {
            action.run();
        } catch (Exception e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            logger.error("Failed to {}: {}", actionName, e);
        }
    }

    private void setRestoreUriIfMissing(StoredResult storedResult, String restoreUri) {
        if (storedResult.getRestoreUri().isPresent()) {
            return;
        }
        String fileId = storedResult.getFileId();
        TargetType targetType = storedResult.getConvertTargetType();
        logger.info("Setting missing restore uri {} for {}, setting it", restoreUri, fileId);
        executeNonCriticalAction("set restore uri",
                () -> storedResultDao.setRestoreUri(fileId, targetType, restoreUri));
    }

    @Override
    public String onSubmit(PassportUidOrZero uid, String reportedContentType, File2 tempFile,
            TargetType convertTargetType, Instant totalStart)
    {
        logger.debug("onSubmit({}, '{}', '{}', {})", uid, reportedContentType, tempFile, convertTargetType);

        FileCopy localCopy = new FileCopy(tempFile);
        boolean isFileUsed = false;
        try {
            final String fileId = digester.calculateDigestId(tempFile);
            final long length = tempFile.length();

            /*
             * We are using special fake URIs so we can use common procedure to
             * start converting
             */
            ActualUri actualUri = new ActualUri("upload://" + fileId);
            uriRightsDao.saveOrUpdateUriRight(actualUri, uid);
            storedUriDao.saveOrUpdateUri(actualUri, Option.of(reportedContentType),
                    convertTargetType, 1f);

            /* Now it will look like we just finished copying from fake URL... */
            CopiedFileInfo info = new CopiedFileInfo(actualUri, Option.of(reportedContentType),
                    Option.empty(), localCopy, fileId, true, Option.empty(),
                    StartConversionInfo.builder()
                            .source(DocumentSourceInfo.builder()
                                    .originalUrl("upload")
                                    .uid(uid)
                                    .build())
                            .isYandex(yandexStaffUserRegistry.isStaffUser(uid)).build(), "fake");
            isFileUsed = onCopyDone(info, "");

            LoggerEventsRecorder.saveSubmitEvent(uid, reportedContentType, length,
                    fileId, TimeUtils.toDurationToNow(totalStart));

            return fileId;
        } finally {
            if (!isFileUsed) {
                localCopy.deleteFileIfPossible();
            }
        }
    }

    // for tests
    public void setStoredResultDao(StoredResultDao storedResultDao) {
        this.storedResultDao = storedResultDao;
    }

    public void setStoredUriDao(StoredUriDao storedUriDao) {
        this.storedUriDao = storedUriDao;
    }

    public void setConvertManager(ConvertManager convertManager) {
        this.convertManager = convertManager;
    }
}
