package ru.yandex.chemodan.app.docviewer.storages.mulca;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.docviewer.log.LoggerEventsRecorder;
import ru.yandex.chemodan.app.docviewer.storages.AbstractFileStorage;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

public class MulcaFileStorage extends AbstractFileStorage {
    static final String SCHEMA = "mulca://";

    @Autowired
    private MulcaClient mulcaClient;

    private String path;

    @Override
    public void delete(FileLink fileLink) {
        MulcaFileLink mulcaFileLink = (MulcaFileLink) fileLink;
        String mulcaId = mulcaFileLink.getMulcaId();
        delete(mulcaId);
    }

    void delete(String mulcaId) {
        try {
            mulcaClient.delete(MulcaId.fromSerializedString(mulcaId));
            LoggerEventsRecorder.saveCleanupFromMulcaEvent(mulcaId);
        } catch (Exception exc) {
            LoggerEventsRecorder.saveCleanupFromMulcaFailedEvent(mulcaId, exc);
        }
    }

    @Override
    public InputStreamSource get(FileLink fileLink) {
        MulcaId mulcaId = MulcaId.fromSerializedString(((MulcaFileLink) fileLink).getMulcaId());
        return RetryUtils.retry(2, () -> mulcaClient.download(mulcaId));
    }

    @Override
    public FileLink put(InputStreamSource inputStreamSource) {
        MulcaId mulcaId = RetryUtils.retry(1,
                () -> mulcaClient.upload(inputStreamSource, "docviewer_" + path, false, true));
        return new MulcaFileLink(mulcaId.toSerializedString());
    }

    public void setPath(String path) {
        this.path = path;
    }

    @Override
    public FileLink toFileLink(String string) {
        Validate.notNull(string, "string == null");

        if (!StringUtils.startsWith(string, SCHEMA))
            throw new IllegalArgumentException("String '" + string
                    + "' is not a valid mulca file link");

        String mulcaId = StringUtils.substringAfter(string, SCHEMA);
        return new MulcaFileLink(mulcaId);
    }
}
