package ru.yandex.chemodan.app.docviewer.utils;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.MessageDigest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoUtils;

public class Digester {

    private static final Logger logger = LoggerFactory.getLogger(Digester.class);

    @Value("${docviewer.digestAlgorithm}")
    private String digestAlgorithm;

    public String calculateDigestId(final InputStreamSource source) {
        logger.debug("Calculating ID for '{}'...", source);

        final long length = FileUtils.calculateLength(source);
        final byte[] digest = calculateDigest(source, digestAlgorithm);
        final String digestStr = new BigInteger(digest).abs().toString(Character.MAX_RADIX);
        final String fileId = Long.toString(length, Character.MAX_RADIX) + "-" + digestStr;

        logger.debug("Calculated ID for '{}': '{}'", source, fileId);
        return fileId;
    }

    private static void calculateDigest(final InputStream source, MessageDigest digest) {
        try {
            final byte[] bytes = new byte[0x100000];
            for (;;) {
                int readCount = source.read(bytes);
                if (readCount < 0)
                    break;

                digest.update(bytes, 0, readCount);
            }
        } catch (IOException exc) {
            throw IoUtils.translate(exc);
        }
    }

    public static byte[] calculateDigest(final InputStreamSource source, String algorithm) {
        try {
            final MessageDigest digest = MessageDigest.getInstance(algorithm);
            source.readNr(inputStream -> calculateDigest(inputStream, digest));
            return digest.digest();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }


    // for tests

    public void setDigestAlgorithm(String digestAlgorithm) {
        this.digestAlgorithm = digestAlgorithm;
    }

}
