package ru.yandex.chemodan.app.docviewer.utils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamXUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.random.Random2;

/**
 * @author vlsergey
 * @author akirakozov
 * @author ssytnik
 */
public class FileUtils {

    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    private static final int TMP_SUBDIR_COUNT = 200;
    private static final AtomicInteger currentSubdirNumber = new AtomicInteger(0);
    private static final File2 rootTmpfsTemporaryDirectory = createRootTemporaryDirectory("/tmpfs", 8);
    private static final File2 rootTemporaryDirectory = rootTmpfsTemporaryDirectory;

    private static File2 createRootTemporaryDirectory(String parentDirectory, int size) {
        try {
            final String prefix = "docviewer";
            final String suffix = ".tmp";
            File file = null;
            if (StringUtils.isNotEmpty(parentDirectory)) {
                File2 rootFiles = new File2(parentDirectory);
                if (rootFiles.exists() && rootFiles.getTotalSpaceSize().ge(DataSize.fromGigaBytes(size))) {
                    logger.info("Enabled: {}, free-space: {}", file, rootFiles.getFreeSpaceSize());
                    file = rootFiles.getFile();
                } else {
                    logger.warn("Used default tmp dir");
                }
            }
            File2 rootTmpDirectory = new File2(File.createTempFile(prefix, suffix, file));
            logger.info("Root tmp directory: {}", rootTmpDirectory);
            boolean deleted = rootTmpDirectory.getFile().delete();
            if (!deleted)
                throw new RuntimeIoException("Unable to delete temporary file: " + rootTmpDirectory);

            rootTmpDirectory.parent().listDirectories()
                    .filter(File2.nameM().andThen(Cf.String.startsWithF(prefix)))
                    .filter(File2.nameM().andThen(Cf.String.endsWithF(suffix)))
                    .forEach(File2.deleteRecursiveQuietlyF());

            mkdir(rootTmpDirectory);
            return rootTmpDirectory;
        } catch (IOException exc) {
            throw IoUtils.translate(exc);
        }
    }

    // sizes

    public static DataSize getTempFilesSize() {
        return rootTemporaryDirectory.usedSpaceRecursive();
    }

    public static DataSize getTotalSpaceSize() {
        return rootTemporaryDirectory.getTotalSpaceSize();
    }

    public static DataSize getTempFreeSize() {
        return rootTemporaryDirectory.getFreeSpaceSize();
    }

    public static DataSize getTempUsableSize() {
        return rootTemporaryDirectory.getUsableSpaceSize();
    }

    public static File2 getRootTempDir(File2 rootTemporaryDirectory) {
        int num = Math.abs(currentSubdirNumber.getAndIncrement() % TMP_SUBDIR_COUNT);
        File2 subdir = rootTemporaryDirectory.child(StringUtils.leftPad(Integer.toString(num), 3, '0'));
        if (!subdir.exists()) {
            mkdir(subdir);
        }
        return subdir;
    }

    public static File2 getRootTempDir() {
        return getRootTempDir(rootTemporaryDirectory);
    }

    public static File2 getParentTempDir() {
        return rootTemporaryDirectory;
    }

    public static File2 getRootTmpfsDir() {
        return getRootTempDir(rootTmpfsTemporaryDirectory);
    }

    // temporary files

    private static File createTempFile(String prefix, String suffix, File dir) {
        File f;
        boolean created;
        do {
            f = new File(dir, prefix + "-" + Random2.R.nextAlnum(18) + suffix);
            try {
                created = f.createNewFile();
            } catch (IOException ex) {
                throw IoUtils.translate(ex);
            }
        } while (!created);
        return f;
    }

    public static File2 createEmptyTempFile(String prefix, String suffix) {
        return createEmptyTempFile(prefix, suffix, getRootTempDir());
    }

    public static File2 createEmptyTempFile(String prefix, String suffix, File2 dir) {
        return new File2(createTempFile(prefix, suffix, dir.getFile()));
    }

    public static File2 createTempFile(String prefix, String suffix, File2 dir,
            InputStreamSource inputStreamSource)
    {
        File2 temp = null;
        try {
            temp = new File2(createTempFile(prefix, suffix, dir.getFile()));
            inputStreamSource.readTo(temp);
            return temp;
        } catch (RuntimeException exc) {
            if (temp != null) {
                temp.deleteRecursiveQuietly();
            }
            throw exc;
        }
    }

    public static File2 createTempFile(String prefix, String suffix, InputStreamSource inputStreamSource) {
        return createTempFile(prefix, suffix, getRootTempDir(), inputStreamSource);
    }

    public static <T> T withEmptyTemporaryFile(String prefix, String suffix, Function<File2, ? extends T> handler) {
        return withEmptyTemporaryFile(prefix, suffix, getRootTmpfsDir(), handler);
    }

    public static <T> T withEmptyTemporaryFile(String prefix, String suffix, File2 rootTmpDir, Function<File2, ? extends T> handler) {
        final File2 temporaryFile = createEmptyTempFile(prefix, suffix, rootTmpDir);
        try {
            return handler.apply(temporaryFile);
        } finally {
            temporaryFile.deleteRecursiveQuietly();
        }
    }

    public static void withEmptyTemporaryFile(String prefix, String suffix, File2 rootTmpDir, Function1V<File2> handler) {
        withEmptyTemporaryFile(prefix, suffix, rootTmpDir, handler.asFunctionReturnNull());
    }

    public static void withEmptyTemporaryFile(String prefix, String suffix, Function1V<File2> handler) {
        withEmptyTemporaryFile(prefix, suffix, handler.asFunctionReturnNull());
    }

    public static void withFile(InputStreamSource inputStreamSource, Function1V<File2> handler) {
        IoUtils.withFile(inputStreamSource, handler.asFunctionReturnNull());
    }

    public static <T> T withFile(OutputStreamSource outputStreamSource, Function<File2, ? extends T> handler) {
        if (outputStreamSource instanceof FileOutputStreamSource) {
            return handler.apply(new File2(((FileOutputStreamSource) outputStreamSource).getFile()));
        }

        File2 temporary = createEmptyTempFile("output", ".tmp");
        boolean deleted = temporary.getFile().delete();
        if (!deleted) {
            logger.warn("Unable to delete temporary file '{}'", temporary);
        }

        try {
            T result = handler.apply(temporary);
            temporary.readTo(outputStreamSource);
            return result;
        } finally {
            temporary.deleteRecursiveQuietly();
        }
    }

    public static void withFile(OutputStreamSource outputStreamSource, Function1V<File2> handler) {
        withFile(outputStreamSource, handler.asFunctionReturnNull());
    }

    public static void withTemporaryFile(String prefix, String suffix,
            InputStreamSource inputStreamSource, Function1V<File2> handler)
    {
        withTemporaryFile(prefix, suffix, inputStreamSource, handler.asFunctionReturnNull());
    }

    public static <T> void withTemporaryFile(String prefix, String suffix,
            InputStreamSource inputStreamSource, Function<File2, ? extends T> handler)
    {
        final File2 temporaryFile = createTempFile(prefix, suffix, getRootTmpfsDir(), inputStreamSource);
        try {
            handler.apply(temporaryFile);
        } finally {
            temporaryFile.deleteRecursiveQuietly();
        }
    }

    // temporary dirs

    public static File2 createTempDirectory(String prefix, String suffix, File2 directory) {
        final File file = createTempFile(prefix, suffix, directory.getFile());

        boolean deleted = file.delete();
        if (!deleted)
            throw new RuntimeIoException("Unable to delete temporary file: " + file);

        boolean created = file.mkdir();
        if (!created)
            throw new RuntimeIoException("Unable to create temporary dir: " + file);

        return new File2(file);
    }

    public static File2 createTempDirectory(String prefix, String suffix) {
        return createTempDirectory(prefix, suffix, rootTemporaryDirectory);
    }

    public static <T> T withTemporaryDirectory(String prefix, String suffix, Function<File2, ? extends T> handler) {
        return withTemporaryDirectory(prefix, suffix, rootTemporaryDirectory, handler);
    }

    public static <T> T withTemporaryDirectory(
            String prefix, String suffix, File2 rootDir,
            Function<File2, ? extends T> handler)
    {
        final File2 temporaryDirectory = createTempDirectory(prefix, suffix, rootDir);
        try {
            return handler.apply(temporaryDirectory);
        } finally {
            temporaryDirectory.deleteRecursiveQuietly();
        }
    }

    // zip

    public static <T> T withZipFile(InputStreamSource inputStreamSource, final Function<ZipFile, ? extends T> handler) {
        return IoUtils.withFile(inputStreamSource, (Function<File2, T>) file -> {
            try {
                final ZipFile zipFile;
                try {
                    zipFile = new ZipFile(file.getFile());
                } catch (ZipException exc) {
                    if ("invalid CEN header (encrypted entry)".equals(exc.getMessage())) {
                        throw new UserException(ErrorCode.FILE_IS_PASSWORD_PROTECTED, exc);
                    } else {
                        throw new RuntimeIoException("Unable to open file '" + file
                                + "' as ZIP archive: " + exc, exc);
                    }
                }
                try {
                    return handler.apply(zipFile);
                } finally {
                    try {
                        zipFile.close();
                    } catch (Throwable exc) {
                        logger.error("Unable to close ZIP file: {}", exc);
                    }
                }
            } catch (IOException exc) {
                throw IoUtils.translate(exc);
            }
        });
    }

    public static void withZipFile(InputStreamSource inputStreamSource, final Function1V<ZipFile> handler) {
        withZipFile(inputStreamSource, handler.asFunctionReturnNull());
    }

    // misc

    /** @deprecated error-prone, use double "with"-constructs */
    public static void copyAndCloseQuitely(InputStream input, OutputStream output) {
        try {
            IoUtils.copy(input, output);
        } finally {
            IoUtils.closeQuietly(input);
            IoUtils.closeQuietly(output);
        }
    }

    // XXX is this really needed? seems we'll get an exception soon enough if any of the conditions fail
    public static void checkFileForReading(File2 file) {
        Validate.isTrue(file.exists() && file.isRegular() && file.getFile().canRead(),
                "File isn't found or can't be read as file: ", file);
    }

    public static long calculateLength(InputStreamSource source) {
        Option<Long> option = source.lengthO();
        if (option.isPresent()) {
            return option.get();
        } else {
            return source.readX(InputStreamXUtils.readToDevNullF());
        }
    }

    private static void mkdir(File2 dir) {
        boolean created = dir.getFile().mkdir();
        if (!created) {
            throw new RuntimeIoException("Unable to create temporary dir: " + dir);
        }
    }
}
