package ru.yandex.chemodan.app.docviewer.utils;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.internet.ContentDisposition;
import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.http.Header;
import org.apache.http.client.ResponseHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.web.framework.FoundException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ReturnHttpCodeException;
import ru.yandex.misc.lang.StringUtils;

public class HttpUtils2 {

    private static final Logger logger = LoggerFactory.getLogger(HttpUtils2.class);

    public static ResponseHandler<String> getFirstHeaderValueResponseHander(final String name) {
        return response -> Option.ofNullable(response.getFirstHeader(name))
                .map(Header::getValue).getOrElse("");
    }

    private static final Pattern filenamePattern = Pattern.compile("^\\w+; filename=(.+)$");

    public static final String HEADER_CONTENT_DISPOSITION = "Content-Disposition";
    public static final String HEADER_LAST_ACCESS = "LastAccess";

    static {
        if (System.getProperty("mail.mime.decodeparameters") == null) {
            System.setProperty("mail.mime.decodeparameters", Boolean.TRUE.toString());
            logger.warn("System property 'mail.mime.decodeparameters' set to 'TRUE'");

        } else {
            logger.warn("System property 'mail.mime.decodeparameters' already set to '{}'",
                    System.getProperty("mail.mime.decodeparameters"));
        }
    }

    public static void cleanupQuietly(List<FileItem> fileItems) {
        for (FileItem fileItem : fileItems) {
            try {
                fileItem.delete();
            } catch (Throwable exc) {
                logger.warn("Unable to delete (cleanup) temporary file: " + exc, exc);
            }
        }
    }

    public static Option<String> getFilenameFromContentDisposition(String contentDispositionHeader)
    {
        if (StringUtils.isEmpty(contentDispositionHeader))
            return Option.empty();

        try {
            ContentDisposition contentDispositionStructure = new ContentDisposition(
                    contentDispositionHeader);

            String fn = contentDispositionStructure.getParameter("filename");
            if (StringUtils.isEmpty(fn)) {
                return Option.empty();
            }

            String filename = recoverUtfValueIfPossible(fn);
            if (StringUtils.isEmpty(filename)) {
                return Option.empty();
            }

            return Option.of(filename);
        } catch (ParseException exc) {
            // attempt to work around mpfs bug
            Matcher m = filenamePattern.matcher(contentDispositionHeader);
            if (m.matches()) {
                try {
                    return Option.of(recoverUtfValueIfPossible(m.group(1)));
                } catch (Exception e) {
                    logger.warn("Error occurred while recovering {}: {}", m.group(1), e.getMessage());
                }
            }
            logger.warn("Unable to parse file name from content disposition header value '"
                    + contentDispositionHeader + "': " + exc, exc);
            return Option.empty();
        }
    }

    public static String recoverUtfValueIfPossible(String s) {
        ListF<Byte> bytes = Cf.arrayList();
        for (char c: s.toCharArray()) {
            if (c > 255) { // string already contains non-ascii characters
                return s;
            }
            bytes.add((byte) c);
        }
        try {
            return new String(bytes.toByteArray(), "UTF8");
        } catch (Exception e) {
            return s;
        }
    }

    public static void sendError(final int statusCode, Throwable exc, HttpServletResponse resp)
            throws IOException
    {
        resp.setStatus(statusCode);

        if (StringUtils.isEmpty(exc.getMessage())) {
            resp.sendError(statusCode);
            return;
        }

        resp.setContentType("text/plain");
        resp.setCharacterEncoding("utf-8");

        PrintWriter writer = resp.getWriter();
        writer.print(exc.getMessage());
        writer.flush();
        writer.close();
    }

    public static void sendError(final ReturnHttpCodeException exc, HttpServletResponse resp)
            throws IOException
    {
        sendError(exc.getStatusCode(), exc, resp);
    }

    public static void sendRedirect(FoundException exc, HttpServletResponse resp)
            throws IOException
    {
        resp.setStatus(exc.getStatusCode());
        final String location = UriUtils.toUrlString(exc.getUri());
        resp.setHeader("Location", location);
        for (Cookie cookie : exc.getCookies()) {
            resp.addCookie(cookie);
        }

        resp.setContentType("text/html");
        resp.setCharacterEncoding("utf-8");

        PrintWriter writer = resp.getWriter();
        writer.print("<html><head><meta http-equiv=\"Refresh\" content=\"0; url="
                + StringEscapeUtils.escapeHtml4(location)
                + "\" /></head><body><p>Please follow <a href=\""
                + StringEscapeUtils.escapeHtml4(location)
                + "\">this link</a>.</p>\n</body>\n</html>");
        writer.flush();
        writer.close();
    }

    public static Option<String> toContentType(String debugLocation, String headerValue) {
        if (StringUtils.isEmpty(headerValue))
            return Option.empty();

        try {
            ContentType contentType = new ContentType(headerValue);
            if (StringUtils.isNotEmpty(contentType.getBaseType()))
                return Option.of(contentType.getBaseType());
        } catch (ParseException exc) {
            logger.debug("Unable to parse content type '" + headerValue + "' from '"
                    + debugLocation + "': " + exc, exc);
        }

        return Option.empty();
    }

}
