package ru.yandex.chemodan.app.docviewer.utils;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.apache.commons.lang3.Validate;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.exec.ProcessUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.TimeUtils;

public class ProcessUtils2 {

    private static final Logger logger = LoggerFactory.getLogger(ProcessUtils2.class);

    private static final Duration aliveCheckPeriod = new Duration(50);

    public static void errorToWarn(final Process process, final String threadName,
            final String prefix, final Logger outputLogger)
    {
        processStreamLines(process.getErrorStream(), threadName, line -> {
            if (!StringUtils.isEmpty(line))
                outputLogger.warn(prefix + line);
        });
    }

    public static Option<Integer> getProcessPidO(Process process) {
        Validate.notNull(process);
        try {
            return Option.of(ProcessUtils.getProcessPid(process));
        } catch (RuntimeIoException exc) {
            return Option.empty();
        }
    }

    public static boolean isAlive(Process process) {
        try {
            process.exitValue();
            return false;
        } catch (IllegalThreadStateException exc) {
            return true;
        }
    }

    public static void killProcess(Process process, Duration softKillTime) {
        if (!isAlive(process)) {
            logger.debug("Not need to kill process - it's dead already (exit code is {})",
                    process.exitValue());
            return;
        }

        Option<Integer> pidOption = getProcessPidO(process);
        if (!pidOption.isPresent()) {
            logger.debug("We don't have process PID. Using destroy() method.");

            final Instant stopAfter = TimeUtils.now().plus(Duration.standardSeconds(10));
            process.destroy();
            while (isAlive(process) && stopAfter.isAfterNow()) {
                ThreadUtils.sleep(aliveCheckPeriod);
                process.destroy();
            }
        } else {
            int pid = pidOption.get();

            try {
                logger.info("Sending TERM signal to process with PID {}...", pid);
                ProcessUtils.kill(pid, "TERM");

                final Instant stopAfter = TimeUtils.now().plus(softKillTime);
                while (isAlive(process) && stopAfter.isAfterNow()) {
                    ThreadUtils.sleep(aliveCheckPeriod);
                }
            } catch (Exception exc) {
                logger.error("Unable to send TERM signal to process with PID " + pid + ": " + exc, exc);
            }

            if (!isAlive(process)) {
                logger.info("Process with PID {} is not alive now. Exit code is {}",
                        pid, process.exitValue());
            } else {
                try {
                    logger.info("Sending KILL signal to process PID {}...", pid);

                    ProcessUtils.kill(pid, "KILL");
                } catch (Exception exc) {
                    logger.error("Unable to kill process with PID " + pid + ": " + exc, exc);
                }
            }
        }
    }

    private static void processStreamLines(final InputStream inputStream, final String threadName,
            final Function1V<String> handler)
    {
        new Thread(threadName) {
            public void run() {
                try {
                    BufferedReader lineNumberReader = new BufferedReader(new InputStreamReader(
                            inputStream));

                    while (true) {
                        String line = lineNumberReader.readLine();

                        if (line == null)
                            return;

                        handler.apply(line);
                    }
                } catch (Exception exc) {
                    logger.error("Exception occured while piping input stream: " + exc, exc);
                }
            }
        }.start();
    }

    public static void stdoutToDebug(final Process process, final String threadName,
            final String prefix, final Logger outputLogger)
    {
        stdoutToDebug(process, threadName, prefix, outputLogger, Option.empty());
    }

    public static void stdoutToDebug(final Process process, final String threadName,
            final String prefix, final Logger outputLogger,
            final Option<Function1V<String>> lineHandler)
    {
        processStreamLines(process.getInputStream(), threadName, line -> {
            if (!StringUtils.isEmpty(line))
                outputLogger.debug(prefix + line);

            if (lineHandler.isPresent())
                lineHandler.get().apply(line);
        });
    }
}
