package ru.yandex.chemodan.app.docviewer.utils;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author akirakozov
 */
public class TmpDirCleanupWorker extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(TmpDirCleanupWorker.class);

    @Value("${tmp.dir.office.files.max.size}")
    private DataSize maxSize;
    @Value("${tmp.dir.office.files.prefix}")
    private String prefix;
    @Value("${tmp.dir.office.files.ttl}")
    private Duration ttl;
    @Value("${tmp.dir.uno.files.ttl}")
    private Duration ttlUno;
    @Value("${tmp.dir.uno}")
    private String tempUno;
    @Value("${tmp.dir.extracted.files.pattern}")    // see ru.yandex.commune.archive.ArchiveManager
    private String extractedFilesPattern;
    @Value("${tmp.dir.extracted.files.ttl}")
    private Duration extractedFilesPatternTtl;

    @Override
    @Value("${tmp.dir.office.cleanup.worker.delay}")
    public void setDelay(Duration delay) {
        super.setDelay(delay);
    }

    @Override
    protected void execute() {
        removeOfficeFiles();
        removeExtractedFiles();
    }

    private void removeOfficeFiles() {
        List<File2> filesToRemove = getOfficeGarbageFiles(Instant.now().minus(ttl), Instant.now().minus(ttlUno));
        if (filesToRemove.isEmpty()) {
            logger.info("Office files: Nothing to remove");
        } else {
            logger.info("Office files: Remove bigger than " + maxSize.toPrettyString());
            for (File2 f : filesToRemove) {
                try {
                    DataSize size = DataSize.fromBytes(f.length());
                    logger.info("Office files: Remove " + f.getAbsolutePath() + ", size: " + size.toPrettyString());
                    f.deleteRecursiveQuietly();
                } catch (Exception e) {
                    logger.warn(e, e);
                }
            }
        }
    }

    private void removeExtractedFiles() {
        List<File2> filesToRemove = getOldExtractedFiles();
        if (filesToRemove.isEmpty()) {
            logger.info("Extracted files: Nothing to remove");
        } else {
            filesToRemove.forEach(f -> {
                try {
                    logger.info("Extracted files: Removing file {}", f);
                    f.deleteRecursiveQuietly();
                } catch (Exception e) {
                    logger.warn(e, e);
                }
            });
        }
    }

    private List<File2> getOldExtractedFiles() {
        long deleteBefore = Instant.now().minus(extractedFilesPatternTtl).getMillis();
        return Cf.x(FileUtils.getTempDirectory().listFiles((file, name) -> name.matches(extractedFilesPattern)))
                .filter(file -> file.lastModified() < deleteBefore).map(File2::new);
    }

    private List<File2> getOfficeGarbageFiles(Instant loOld, Instant unoOld) {
        ListF<File2> temp = Cf.x(ru.yandex.chemodan.app.docviewer.utils.FileUtils.getParentTempDir()
                .listDirectories().stream()
                .filter(d -> d.getName().startsWith(prefix))
                .filter(dir -> DataSize.fromBytes(FileUtils.sizeOfDirectory(dir.getFile())).gt(maxSize) ||
                        dir.lastModified().isBefore(loOld))
                .collect(Collectors.toList()));
        return temp.plus(new File2(tempUno)
                .listDirectories().stream()
                .filter(dir -> dir.lastModified().isBefore(unoOld))
                .collect(Collectors.toList()));
    }
}
