package ru.yandex.chemodan.app.docviewer.utils;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.net.uri.Uri2;

public class UriUtils {

    /*
     * This method is called only if content disposition filename is missing
     */
    public static String getFileName(String originalUrl) {
        String filename = StringUtils.substringBefore(originalUrl, "#");
        filename = StringUtils.substringBefore(filename, "?");
        filename = StringUtils.substringAfterLast("/" + filename, "/");
        return filename;
    }

    public static File2 toFile(URL url) {
        Validate.notNull(url, "url");

        return new File2(new File(toUri(url)));
    }

    public static URI toUri(String uri) {
        Validate.notNull(uri, "uri");

        try {
            return new URI(uri);
        } catch (URISyntaxException exc) {
            throw new RuntimeURISyntaxException(uri, exc);
        }
    }

    public static URI toUri(URL url) {
        Validate.notNull(url, "url");

        try {
            return url.toURI();
        } catch (URISyntaxException exc) {
            throw new RuntimeURISyntaxException(url, exc);
        }
    }

    // http://tools.ietf.org/html/rfc2396#section-2.4.3
    private static SetF<Character> URI_EXCLUDED_US_ASCII = Cf.set(
            ' ',
            // don't escape percent https://jira.yandex-team.ru/browse/DOCVIEWER-261
            '<', '>', '#', /*'%',*/ '"',
            '{', '}', '|', '\\', '^', '[', ']', '`'
            );

    /**
     * @deprecated XXX ssytnik 20-01-2012: wrong:
     * 1) encoded url cannot contain non-encoded percent sign;
     * 2) the need of of '#' escaping is actually ambiguous.
     * Thus, we should not allow just-in-case encoding of input url.
     * Instead, we should accept only valid, properly escaped, url-s.
     */
    public static String escapeUriExcludedUsAscii(String s) {
        StringBuilder sb = new StringBuilder();
        for (char c : StringUtils.characters(s)) {
            if (URI_EXCLUDED_US_ASCII.containsTs(c)) {
                sb.append("%").append(Hex.encode(new byte[] {(byte) c}));
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    public static URI toUriSafe(String externalUrl) {
        return UrlUtils.uri(escapeUriExcludedUsAscii(externalUrl));
    }

    public static String toUriString(File2 file) {
        Validate.notNull(file, "file");

        return file.getFile().toURI().toString();
    }

    public static String toUriString(URI uri) {
        Validate.notNull(uri, "uri");

        return uri.toString();
    }

    public static URL toUrl(File2 file) {
        Validate.notNull(file, "file");

        return toUrl(file.getFile().toURI());
    }

    public static URL toUrl(String url) {
        Validate.notNull(url, "url");

        try {
            return new URL(url);
        } catch (MalformedURLException exc) {
            throw new RuntimeMalformedURLException(url, exc);
        }
    }

    public static URL toUrl(URI uri) {
        Validate.notNull(uri, "uri");

        try {
            return uri.toURL();
        } catch (Exception exc) {
            throw new RuntimeMalformedURLException(uri, exc);
        }
    }

    public static String toUrlString(File2 file) {
        Validate.notNull(file, "file");

        return toUrl(file).toString();
    }

    public static String toUrlString(URI uri) {
        Validate.notNull(uri, "uri");

        return toUrl(uri).toString();
    }

    public static String toUrlString(URL url) {
        Validate.notNull(url, "url");

        return url.toString();
    }

    public static URI urlToUri(String url) {
        Validate.notNull(url, "url");

        return toUri(toUrl(url));
    }

    public static Option<String> getHostO(URI uri) {
        return Uri2.parse(uri.toString()).getHost();
    }

    public static String getHost(URI uri) {
        return UriUtils.getHostO(uri).getOrThrow("Empty host, url: " + uri);
    }
}
