package ru.yandex.chemodan.app.docviewer.utils.cache;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;

import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.Striped;
import lombok.val;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.utils.AbstractCache;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.cache.impl.WeightedLruCache;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author vlsergey
 * @author akirakozov
 */
public class TemporaryFileCache extends AbstractCache {

    private static final Logger logger = LoggerFactory.getLogger(TemporaryFileCache.class);

    private final TemporaryFilesCleaner temporaryFilesCleaner;
    private final Striped<Lock> rwLockStripes = Striped.lock(24);

    private final DynamicProperty<Double> maxAllocationTmpRate =
            new DynamicProperty<>("docviewer.max.allocation.tmp.rate", 0.5d);

    private final class FileLruCache extends WeightedLruCache<FileLink, TempFileItem> {

        private FileLruCache(long maxSize) {
            super(maxSize, TempFileItem::getFileSize);
        }

        @Override
        protected void onRemove(FileLink key, TempFileItem value) {
            super.onRemove(key, value);
            if (destroyed.get()) {
                value.getFile().deleteRecursiveQuietly();
            } else {
                temporaryFilesCleaner.addFileToCleanup(value);
            }
            logger.debug("File was removed from temporary file cache: " + value.getFile());
        }
    }

    private final FileLruCache cache;

    private final AtomicBoolean destroyed = new AtomicBoolean(false);

    public TemporaryFileCache(
            TemporaryFilesCleaner temporaryFilesCleaner,
            DataSize maxCacheSize)
    {
        long min = Math.min(Math.round(FileUtils.getTotalSpaceSize().toBytes() * maxAllocationTmpRate.get()), maxCacheSize.toBytes());
        this.cache = new FileLruCache(min);
        this.temporaryFilesCleaner = temporaryFilesCleaner;
        temporaryFilesCleaner.setTemporaryFileCache(this);
    }

    @PreDestroy
    public void destroy() {
        destroyed.set(true);

            List<FileLink> keys = cache.getKeysOldestToNewest();
            for (FileLink fileLink : keys) {
                cache.removeFromCache(fileLink);
            }
    }

    public Optional<TempFileItem> getInternal(final FileLink fileLink) {
        Validate.isTrue(!destroyed.get(), "Application is stopping");

        onCacheCall();
        Optional<TempFileItem> option = cache.getFromCache(fileLink);
        if (option.isPresent()) {
            TempFileItem item = option.get();
            File2 file = item.getFile();
            if (file.exists() && file.isRegular() && file.getFile().canRead()) {
                item.updateLastUsage();
                return option;
            } else {
                cache.removeFromCache(fileLink);
                return Optional.empty();
            }
        } else {
            onCacheMiss();
            return option;
        }
    }

    public Option<File2> get(final FileLink fileLink) {
        Optional<TempFileItem> internal = getInternal(fileLink);
        return Option.x(internal.map(TempFileItem::getFile));
    }

    public File2 getOrCreateTemporaryFile(final FileLink fileLink, final Function<FileLink, File2> fileSource) {
        Stopwatch stopWatch = Stopwatch.createAndStart();
        try {
            return getInternal(fileLink).map(TempFileItem::getFile).orElseGet(() -> {
                val start = System.currentTimeMillis();
                logger.info("cache miss, writing new file");
                File2 file = fileSource.apply(fileLink);
                logger.info("wrote {} bytes in {}ms", file.length(), System.currentTimeMillis() - start);
                return putInCacheInternal(fileLink, file);
            });
        } finally {
            stopWatch.stopAndLog("getOrCreateTemporaryFile()", logger);
        }
    }

    public File2 putInCache(FileLink fileLink, File2 temporaryFile) {
        Validate.isTrue(!destroyed.get(), "Application is stopping");
        return putInCacheInternal(fileLink, temporaryFile);
    }

    private File2 putInCacheInternal(FileLink fileLink, File2 temporaryFile) {
        Stopwatch stopWatch = Stopwatch.createAndStart();
        Lock lock = rwLockStripes.get(fileLink);
        lock.lock();
        try {
            stopWatch.log("Enter critical section of putInCacheInternal", logger);
            Optional<TempFileItem> cachedFile = getInternal(fileLink);
            if (cachedFile.isPresent()) {
                stopWatch.stopAndLog(
                        "File " + fileLink + "' was already found in temporary files cache: "
                                + cachedFile.get().getFile() + "', remove new one " + temporaryFile, logger);
                temporaryFile.deleteRecursiveQuietly();
                return cachedFile.get().getFile();
            } else {
                cache.putInCache(fileLink, new TempFileItem(temporaryFile));
                stopWatch.stopAndLog(
                        "File " + fileLink + " was added to temporary files cache: " + temporaryFile, logger);
                return temporaryFile;
            }
        } finally {
            lock.unlock();
        }
    }

    public void removeFromCache(FileLink fileLink) {
        Lock lock = rwLockStripes.get(fileLink);
        lock.lock();
        try {
            cache.removeFromCache(fileLink);
        } finally {
            lock.unlock();
        }
    }
}
