package ru.yandex.chemodan.app.docviewer.utils.cache;

import java.util.Comparator;
import java.util.PriorityQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.PreDestroy;

import lombok.Data;
import org.apache.commons.io.filefilter.AgeFileFilter;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.docviewer.storages.fsstorage.FileSystemFileLink;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author akirakozov
 */
@Data
public class TemporaryFilesCleaner extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(TemporaryFilesCleaner.class);

    private final PriorityQueue<TempFileItem> queue = new PriorityQueue<>(
            10, Comparator.comparing(TempFileItem::getLastUsage));

    private final AtomicBoolean destroyed = new AtomicBoolean(false);
    private final Executor executor = Executors.newCachedThreadPool(new ThreadNameIndexThreadFactory("force-clean"));

    private final Duration tempFileTtl;
    private final int forceDeleteAfterAttempts;
    private final DataSize warnFreeSpace;
    private final DataSize critFreeSpace;
    private final String soPrefix;
    private TemporaryFileCache temporaryFileCache;

    public TemporaryFilesCleaner(Duration tempFileTtl, Duration delay, DataSize warnFreeSpace, DataSize critFreeSpace,
                                 int forceDeleteAfterAttempts, String soPrefix) {
        this.tempFileTtl = tempFileTtl;
        this.warnFreeSpace = warnFreeSpace;
        this.critFreeSpace = critFreeSpace;
        this.forceDeleteAfterAttempts = forceDeleteAfterAttempts;
        this.soPrefix = soPrefix;
        setDelay(delay);
    }

    @Override
    protected void execute() {
        ListF<TempFileItem> fileItems = getFilesToRemove();
        for (TempFileItem fileItem : fileItems) {
            logger.debug("Remove file from hdd: " + fileItem.getFile());
            try {
                fileItem.getFile().deleteRecursiveQuietly();
            } finally {
                if (fileItem.getFile().exists()) {
                    logger.debug("File wasn't removed, add it to queue again: " + fileItem.getFile());
                    addFileToCleanup(fileItem);
                } else {
                    logger.debug("File was successfully removed: " + fileItem.getFile());
                }
            }
        }
        cleanup();
    }

    private void cleanup() {
        long created = Instant.now().minus(tempFileTtl).getMillis();
        FileUtils.getParentTempDir().listDirectories()
                .filter(f -> !f.getName().startsWith(soPrefix))
                .map(parent -> CompletableFuture.supplyAsync(() -> {
                    org.apache.commons.io.FileUtils
                            .iterateFilesAndDirs(parent.getFile(),
                                    new AgeFileFilter(created),
                                    new AgeFileFilter(created))
                            .forEachRemaining(file -> {
                                try {
                                    File2 next = new File2(file);
                                    if (check(next.lastModified())) {
                                        if (next.isDirectory()) {
                                            cleanDir(next);
                                        } else {
                                            cleanFile(next);
                                        }
                                    }
                                } catch (Exception e) {
                                    logger.warn("Can't clean {}", e);
                                }
                            });
                    return null;
                }, executor));
    }

    private void cleanFile(File2 next) {
        logger.info("About to clean outdated file {}, {}",
                next, next.length());
        temporaryFileCache.removeFromCache(new FileSystemFileLink(next.getFile()));
        next.deleteRecursiveQuietly();
    }

    private void cleanDir(File2 next) {
        if (next.list().isEmpty() && !FileUtils.getParentTempDir().equals(next)) {
            logger.info("About to clean outdated empty dir {}, {}",
                    next, next.length());
            next.deleteRecursiveQuietly();
        }
    }

    private boolean check(Instant lastModified) {
        if (checkCritLowSpace()) {
            return true;
        } else {
            return lastModified.plus(tempFileTtl.multipliedBy(forceDeleteAfterAttempts)).isBeforeNow();
        }
    }

    public boolean checkWarnLowSpace() {
        DataSize tempFreeSize = FileUtils.getTempFreeSize();
        return tempFreeSize.lt(warnFreeSpace);
    }

    public boolean checkCritLowSpace() {
        DataSize tempFreeSize = FileUtils.getTempFreeSize();
        if (tempFreeSize.lt(critFreeSpace)) {
            logger.warn(
                    "critically low free space: {} of required {} is available",
                    tempFreeSize.toPrettyString(),
                    critFreeSpace.toPrettyString()
            );
            return true;
        } else {
            return false;
        }
    }

    private synchronized ListF<TempFileItem> getFilesToRemove() {
        ListF<TempFileItem> filesToRemove = Cf.arrayList();
        while (!queue.isEmpty() && (checkWarnLowSpace() ||
                queue.element().getLastUsage().plus(tempFileTtl).isBeforeNow())) {
            filesToRemove.add(queue.poll());
        }
        return filesToRemove;
    }

    public synchronized void addFileToCleanup(TempFileItem fileItem) {
        if (!destroyed.get()) {
            queue.add(fileItem);
        } else {
            throw new IllegalStateException("Application shutdown");
        }
    }

    @PreDestroy
    public void destroy() {
        destroyed.set(true);

        while (!queue.isEmpty()) {
            queue.poll().getFile().deleteRecursiveQuietly();
        }
    }
}
