package ru.yandex.chemodan.app.docviewer.utils.html;

import java.io.BufferedOutputStream;
import java.util.List;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.dom4j.Document;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.XmlSerializer;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.time.TimeUtils;

public class ConvertToHtmlHelper {

    private static final Logger logger = LoggerFactory.getLogger(ConvertToHtmlHelper.class);

    public static final String ZIPENTRY_NAME_RESULT_HTML = "result.html";

    public static final String ZIPENTRY_NAME_RESULT_XML = "result.xml";

    @Autowired
    private HtmlPostprocessor htmlPostprocessor;

    @Autowired
    private HtmlSerializer htmlSerializer;

    @Autowired
    private HtmlSplitter htmlSplitter;

    @Autowired
    private XmlSerializer xmlSerializer;

    /**
     * @return number of result HTML pages
     */
    public int splitAndPack(final Document resultHtml, final OutputStreamSource result,
            boolean filterHtml, final Option<FileList> images, boolean isText)
    {
        return splitAndPack(resultHtml, result,
                () -> split(resultHtml, filterHtml, images, isText));
    }

    public int splitAndPack(final Document resultHtml, final OutputStreamSource result,
            boolean filterHtml, final Option<FileList> images, DataSize maxSizePart, DataSize maxSizeSingle)
    {
        return splitAndPack(resultHtml, result,
                () -> split(resultHtml, filterHtml, images, maxSizePart, maxSizeSingle));
    }

    public int splitAndPack(final Document resultHtml, final OutputStreamSource result,
            Function0<List<byte[]>> splitFunction)
    {
        ThreadLocalTimeout.check();
        Instant start = TimeUtils.now();

        final List<byte[]> pages = splitFunction.apply();

        logger.debug("Split into {} page(s) in {}", pages.size(), TimeUtils.toDurationToNow(start));
        start = TimeUtils.now();

        pack(resultHtml, result, pages);
        logger.debug("Compressed into ZIP in {}", TimeUtils.toDurationToNow(start));
        return pages.size();
    }

    private List<byte[]> split(Document resultHtml, boolean filterHtml, Option<FileList> images, boolean isText) {
        return htmlSplitter.splitHtml(resultHtml, filterHtml, images, isText);
    }

    private List<byte[]> split(Document resultHtml, boolean filterHtml, Option<FileList> images,
            DataSize maxSizePart, DataSize maxSizeSingle)
    {
        return htmlSplitter.splitHtml(resultHtml, filterHtml, images, maxSizePart, maxSizeSingle);
    }

    private void pack(Document resultHtml, OutputStreamSource result, List<byte[]> pages) {
        result.writeBuffered((IoFunction1V<BufferedOutputStream>) outputStream -> {
            ZipOutputStream zipOutputStream = new ZipOutputStream(outputStream);
            try {
                zipOutputStream.setLevel(Deflater.BEST_SPEED);

                for (int p = 0; p < pages.size(); p++) {
                    zipOutputStream.putNextEntry(new ZipEntry("page_" + (p + 1) + ".xml"));
                    zipOutputStream.write(pages.get(p));
                }

                zipOutputStream.putNextEntry(new ZipEntry(ZIPENTRY_NAME_RESULT_XML));
                xmlSerializer.serializeToXml(resultHtml, zipOutputStream);

                zipOutputStream.putNextEntry(new ZipEntry(ZIPENTRY_NAME_RESULT_HTML));
                htmlPostprocessor.preprocessOutput(resultHtml, Option.empty(),
                        DimensionO.WIDTH_1024);
                htmlSerializer.serializeToHtml(resultHtml, zipOutputStream);

            } finally {
                IoUtils.closeQuietly(zipOutputStream);
            }
        });
    }
}
