package ru.yandex.chemodan.app.docviewer.utils.html;

import java.net.URL;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hwpf.converter.HtmlDocumentFacade;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.chemodan.app.docviewer.web.backend.HtmlImageAction;
import ru.yandex.misc.io.http.UrlUtils;

public class HtmlPostprocessor {

    private static final String X_DOCVIEWER_IMG_LOCAL_SRC = "x-docviewer-img-local-src";
    private static final String X_DOCVIEWER_IMG_SINGLE = "x-docviewer-img-single";
    private static final String X_DOCVIEWER_IMG_PLACEHOLDER = "x-docviewer-img-placeholder";
    private static final String X_DOCVIEWER_PAGE_BACKGROUND = "x-docviewer-page-background";
    private static final String X_DOCVIEWER_PAGE_NUMBER_REF = "x-docviewer-page-num-ref";

    private static final Pattern IMG_SRC_ATTRIBUTE_PAGE_BG_PATTERN = Pattern.compile(
            Pattern.quote(X_DOCVIEWER_PAGE_BACKGROUND) + "-(\\d+)");

    public static final URL IMAGE_PLACEHOLDER_URL = HtmlPostprocessor.class.getResource("img-placeholder.png");

    @Autowired
    private PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;

    public void preprocessOutput(final Document pageDocument, final Option<String> fileId, final DimensionO size) {
        String ext = pdfRenderTargetTypeHolder.getTargetType().value();

        Option<Element> headOption = HtmlUtils.findHead(pageDocument);
        if (headOption.isPresent()) {
            Element head = headOption.get();
            for (Element style : HtmlUtils.findChildElements(head, "style")) {
                String text = style.getTextTrim();
                // DOM4J bug workaround
                style.setText(text.length() == 0 ? " " : text);
            }
        }

        Option<Element> bodyO = HtmlUtils.findBody(pageDocument);
        if (bodyO.isPresent()) {
            Element body = bodyO.get();
            for (Element pageDiv : HtmlUtils.findChildElements(body, "div")) {
                for (Element img : HtmlUtils.findChildElements(pageDiv, "img")) {
                    Attribute src = img.attribute("src");
                    if (src != null) {
                        String text = src.getText();
                        Matcher matcher = IMG_SRC_ATTRIBUTE_PAGE_BG_PATTERN.matcher(text);
                        if (matcher.find()) {
                            text = matcher.replaceAll(getPathReplacement(fileId, size, ext));
                        }
                        src.setText(text);
                    }
                }
            }
        }

        new AbstractDom4jVisitor() {
            @Override
            protected void visit(final Element element) {
                if (StringUtils.equalsIgnoreCase("img", element.getName())) {
                    withRemovedAttribute(element, X_DOCVIEWER_IMG_PLACEHOLDER, value -> element.addAttribute("src", fileId.isPresent() ?
                            HtmlImageAction.PATH + "?placeholder=true" : "no-image.png"));

                    withRemovedAttribute(element, X_DOCVIEWER_IMG_LOCAL_SRC, value -> element.addAttribute("src", fileId.isPresent() ?
                            HtmlImageAction.PATH + "?id=" + fileId.get() + "&name=" + value : value));

                    withRemovedAttribute(element, X_DOCVIEWER_IMG_SINGLE, value -> {
                        if (size.width.isPresent()) {
                            element.addAttribute("width", String.valueOf(size.width.get()));
                        }
                        if (size.height.isPresent()) {
                            element.addAttribute("height", String.valueOf(size.height.get()));
                        }
                    });
                }
            }
        }.visit(pageDocument);
    }

    private static void withRemovedAttribute(Element element, String attributeName, Function1V<String> f) {
        Attribute attribute = element.attribute(attributeName);
        if (attribute != null) {
            f.apply(attribute.getValue());
            element.remove(attribute);
        }
    }

    private static String getPathReplacement(Option<String> fileId, DimensionO size, String ext) {
        if (fileId.isPresent()) {
            return HtmlImageAction.PATH
                    + "?id=" + UrlUtils.urlEncode(fileId.get())
                    + (size.width.isPresent() ? "&width=" + size.width.get() : "")
                    + (size.height.isPresent() ? "&height=" + size.height.get() : "")
                    + "&name=bg-$1." + ext;
        } else {
            return "bg-$1." + ext;
        }
    }


    public static void addLocalImageSrc(Element img, String name, boolean isSingle) {
        img.addAttribute(X_DOCVIEWER_IMG_LOCAL_SRC, name);
        if (isSingle) {
            img.addAttribute(X_DOCVIEWER_IMG_SINGLE, "true");
        }
    }

    public static Option<String> readLocalImageSrc(Element img) {
        return Option.ofNullable(img.attribute(X_DOCVIEWER_IMG_LOCAL_SRC)).map(Attribute::getValue);
    }

    public static void addImagePlaceHolder(Element img) {
        img.addAttribute(X_DOCVIEWER_IMG_PLACEHOLDER, "true");
    }

    public static void addPageNumberRef(Element element, int page) {
        element.addAttribute(X_DOCVIEWER_PAGE_NUMBER_REF, Integer.toString(page));
    }

    public void addPageBackground(HtmlDocumentFacade htmlDocumentFacade,
            org.w3c.dom.Element pageDiv, int pageIndex, int htmlWidth, int htmlHeight)
    {
        org.w3c.dom.Element img = htmlDocumentFacade.createImage(X_DOCVIEWER_PAGE_BACKGROUND + "-" + pageIndex);
        img.setAttribute("width", String.valueOf(htmlWidth));
        img.setAttribute("height", String.valueOf(htmlHeight));
        pageDiv.appendChild(img);
    }
}
