package ru.yandex.chemodan.app.docviewer.utils.html;

import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.PostConstruct;

import lombok.Data;
import org.dom4j.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.xml.stream.XmlWriterFactory;

@Data
public class HtmlSplitter {

    @Value("${html.localimagepattern}")
    private Pattern localImagePattern;

    @Value("${html.partsize}")
    private DataSize maxSizePart;
    @Value("${html.singlepartsize}")
    private DataSize maxSizeSingle;

    @Value("${html.partsize.text}")
    private DataSize maxSizePartText;
    @Value("${html.singlepartsize.text}")
    private DataSize maxSizeSingleText;

    @Value("${xml.prefer.default.format}")
    private boolean xmlPreferDefaultFormat;

    private XmlWriterFactory xmlWriterFactory;

    @Autowired
    private PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;

    private Pattern filenameBackgroundPattern;

    @PostConstruct
    public void afterPropertiesSet() {
        String ext = pdfRenderTargetTypeHolder.getTargetType().value();
        filenameBackgroundPattern = Pattern.compile("^bg\\-([0-9]+)\\." + ext +"$");
        xmlWriterFactory = XmlWriterFactory.staxStream();
    }

    public DataSize getMaxSizePart() {
        return maxSizePart;
    }

    public DataSize getMaxSizeSingle() {
        return maxSizeSingle;
    }

    public DataSize getMaxSizePartText() {
        return maxSizePartText;
    }

    public DataSize getMaxSizeSingleText() {
        return maxSizeSingleText;
    }

    public void setLocalImagePattern(Pattern localImagePattern) {
        this.localImagePattern = localImagePattern;
    }

    public void setMaxSizePart(DataSize maxSizePart) {
        this.maxSizePart = maxSizePart;
    }

    public void setMaxSizeSingle(DataSize maxSizeSingle) {
        this.maxSizeSingle = maxSizeSingle;
    }

    public void setMaxSizePartText(DataSize maxSizePartText) {
        this.maxSizePartText = maxSizePartText;
    }

    public void setMaxSizeSingleText(DataSize maxSizeSingleText) {
        this.maxSizeSingleText = maxSizeSingleText;
    }

    public List<byte[]> splitHtml(Document sourceHtml, boolean filter, Option<FileList> images, boolean isText) {
        return splitHtml(sourceHtml, filter, images,
                (isText ? maxSizePartText : maxSizePart),
                (isText ? maxSizeSingleText : maxSizeSingle));
    }

    public List<byte[]> splitHtml(Document sourceHtml, boolean filter, Option<FileList> images,
            DataSize maxSizePart, DataSize maxSizeSingle)
    {
        if (!images.isPresent()) {
            new RemoveImagesProcessor().visit(sourceHtml);
        }

        if (filter) {
            new HtmlFilter(sourceHtml).process();
        }

        new HtmlTablesWidthProcessor().visit(sourceHtml);
        new HtmlMinWidthProcessor().visit(sourceHtml);

        HyperlinkProcessor hyperlinkProcessor = new HyperlinkProcessor(images);
        hyperlinkProcessor.visit(sourceHtml);

        HtmlSplitterWorker worker = new HtmlSplitterWorker(sourceHtml,
                xmlWriterFactory, xmlPreferDefaultFormat, "utf-8",
                maxSizePart,
                maxSizeSingle,
                hyperlinkProcessor.hasAnchors());

        return worker.splitHtml();
    }

    public Pattern getFilenameBackgroundPattern() {
        return filenameBackgroundPattern;
    }

}
