package ru.yandex.chemodan.app.docviewer.utils.pdf;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.DoubleUnaryOperator;

import org.apache.pdfbox.cos.COSArray;
import org.apache.pdfbox.cos.COSBase;
import org.apache.pdfbox.cos.COSStream;
import org.apache.pdfbox.cos.COSString;
import org.apache.pdfbox.io.RandomAccessFile;
import org.apache.pdfbox.pdfwriter.ContentStreamWriter;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDDocumentInformation;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.common.PDStream;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.pdmodel.font.PDType0Font;
import org.apache.pdfbox.util.PDFOperator;
import org.apache.pdfbox.util.PDFStreamEngine;
import org.apache.pdfbox.util.PDFTextStripper;
import org.apache.pdfbox.util.ResourceLoader;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function3B;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.convert.result.PageInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Stopwatch;

public class PdfUtils {

    private static final Logger logger = LoggerFactory.getLogger(PdfUtils.class);

    private static class PdfCleaner extends PDFStreamEngine {
        private int level = 0;

        private final List<Object> newTokens;

        private final Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction;

        private PdfCleaner(List<Object> newTokens,
                Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction)
                throws IOException
        {
            super(ResourceLoader.loadProperties("ru/yandex/chemodan/app/docviewer/utils/PdfCleaner.properties",
                    true));
            this.newTokens = newTokens;
            this.preserveFunction = preserveFunction;
        }

        @Override
        protected void processOperator(PDFOperator operator, List<COSBase> arguments)
                throws IOException
        {
            level++;
            try {
                super.processOperator(operator, arguments);
                if (level == 1) {
                    if (preserveFunction.apply(this, operator, arguments)) {
                        newTokens.addAll(arguments);
                        newTokens.add(operator);
                    }
                }
            } finally {
                level--;
            }
        }

        void processPage(PDPage page) throws IOException {
            resetEngine();
            PDStream contentStream = page.getContents();

            if (contentStream == null)
                return;

            COSStream content = contentStream.getStream();
            processStream(page, page.findResources(), content);
        }

    }

    public static DocumentProperties getProperties(PDDocument pdDocument) {
        DocumentProperties properties = DocumentProperties.EMPTY;
        PDDocumentInformation info = pdDocument.getDocumentInformation();
        if (info != null) {
            properties = properties.withProperty(DocumentProperties.AUTHOR, info.getAuthor());
            properties = properties.withProperty(DocumentProperties.CREATOR, info.getCreator());
            properties = properties.withProperty(DocumentProperties.KEYWORDS, info.getKeywords());
            properties = properties.withProperty(DocumentProperties.SUBJECT, info.getSubject());
            properties = properties.withProperty(DocumentProperties.TITLE, info.getTitle());
        }
        return properties;
    }

    private static void closeQuietly(PDDocument pdDocument) {
        try {
            pdDocument.close();
        } catch (Throwable exc) {
            logger.error("Unable to close PDDocument: " + exc, exc);
        }
    }

    public static void filterText(InputStreamSource pdfSource, int oneBasedPageIndex,
            OutputStreamSource pdfOutput)
    {
        PdfUtils.withExistingDocument(
                pdfSource,
                true,
                PdfUtils.filterTextOperatorsHandler(PdfUtils.preserveInBackgroundHandler(true),
                        oneBasedPageIndex, oneBasedPageIndex).asFunctionReturnParam()
                        .andThen(PdfUtils.writeHandler(pdfOutput)));
    }

    public static void filterTextOperators(PDDocument document,
            final Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction)
    {
        filterTextOperators(document, preserveFunction, 1, Integer.MAX_VALUE);
    }

    @SuppressWarnings("unchecked")
    private static void filterTextOperators(PDDocument document,
            final Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction,
            int oneBasedStartPage, int oneBasedEndPage)
    {
        try {
            List<PDPage> allPages = document.getDocumentCatalog().getAllPages();
            for (int i = oneBasedStartPage; i <= Math.min(oneBasedEndPage, allPages.size()); i++) {
                PDPage pdPage = allPages.get(i - 1);

                final List<Object> newTokens = new ArrayList<>();

                final PdfCleaner engine = new PdfCleaner(newTokens, preserveFunction);
                engine.processPage(pdPage);

                PDStream newContents = new PDStream(document);
                ContentStreamWriter writer = new ContentStreamWriter(
                        newContents.createOutputStream());
                writer.writeTokens(newTokens);
                pdPage.setContents(newContents);
            }
        } catch (Exception exc) {
            throw ExceptionUtils.throwException(exc);
        }
    }

    public static Function1V<PDDocument> filterTextOperatorsHandler(
            final Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction)
    {
        return a -> filterTextOperators(a, preserveFunction);
    }

    private static Function1V<PDDocument> filterTextOperatorsHandler(
            final Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveFunction,
            final int oneBasedStartPage, final int oneBasedEndPage)
    {
        return document -> filterTextOperators(document, preserveFunction, oneBasedStartPage, oneBasedEndPage);
    }

    public static void getSinglePageInfo(PDDocument document, int oneBasedPageIndex) {
        getSinglePageInfo(getPage(document, oneBasedPageIndex), oneBasedPageIndex);
    }

    public static PageInfo getSinglePageInfo(PDPage page, int oneBasedPageIndex) {
        final PDRectangle pageMediaBox = page.findMediaBox();
        final PDRectangle pageCropBox = page.findCropBox();

        final float viewLeft = Math.max(pageMediaBox.getLowerLeftX(), pageCropBox.getLowerLeftX());
        final float viewBottom = Math.max(pageMediaBox.getLowerLeftY(), pageCropBox.getLowerLeftY());
        final float viewTop = Math.min(pageMediaBox.getUpperRightY(), pageCropBox.getUpperRightY());
        final float viewRight = Math.min(pageMediaBox.getUpperRightX(), pageCropBox.getUpperRightX());

        final float viewWidth;
        final float viewHeight;

        final int rotation = page.findRotation();
        if (rotation == 90 || rotation == 270) {
            viewHeight = viewRight - viewLeft;
            viewWidth = viewTop - viewBottom;
        } else {
            viewWidth = viewRight - viewLeft;
            viewHeight = viewTop - viewBottom;
        }

        return new PageInfo(oneBasedPageIndex, viewWidth, viewHeight);
    }


    private static PDPage getPage(PDDocument document, int oneBasedPageIndex) {
        return (PDPage) document.getDocumentCatalog().getAllPages().get(oneBasedPageIndex - 1);
    }

    private static boolean isSingleSpecialMathChar(PDFont font, byte[] string) {
        try {
            if (string.length == 1) {
                int codeLength;
                for (int i = 0; i < string.length; i += codeLength) {
                    // Decode the value to a Unicode character
                    codeLength = 1;
                    String c = font.encode(string, i, codeLength);
                    if (c == null && i + 1 < string.length) {
                        // maybe a multibyte encoding
                        codeLength++;
                        c = font.encode(string, i, codeLength);
                    }
                    if (c == null)
                        return false;
                    if (c.length() == 2 && (c.charAt(1) == '\ufe00' || c.charAt(1) == '\ufe02')) {
                        return true;
                    }
                    if (StringUtils.containsAny(c, "\u20e6\u239b\u239c\u239d\u239e\u239f\u23a0")) {
                        return true;
                    }
                }
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
        return false;
    }

    public static PDDocument load(InputStreamSource inputStreamSource, File2 scratchFile, boolean force) {
        logger.debug("Loading PDF from '{}' using scratch file '{}' (force = {})", inputStreamSource, scratchFile,
                force);

        return loadImpl(inputStreamSource, scratchFile, force);
    }

    private static PDDocument loadImpl(final InputStreamSource inputStreamSource,
            final File2 scratchFile, final boolean force)
    {
        return inputStreamSource.readX(
                (IoFunction<InputStreamX, PDDocument>) inputStream -> PDDocument.load(inputStream, newRandomAccessFile(scratchFile), force));
    }

    private static RandomAccessFile newRandomAccessFile(final File2 scratchFile) {
        try {
            return new RandomAccessFile(scratchFile.getFile(), "rw");
        } catch (FileNotFoundException exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    public static Function3B<PDFStreamEngine, PDFOperator, List<COSBase>>
            preserveInBackgroundHandler(final boolean considerComplexMath)
    {
        return (engine, operator, arguments) -> {
            final String operation = operator.getOperation();
            if ("TJ".equals(operation) || "Tj".equals(operation) || "'".equals(operation)
                    || "\"".equals(operation))
            {
                PDFont font = engine.getGraphicsState().getTextState().getFont();
                if (font instanceof PDType0Font) {
                    PDType0Font type0Font = (PDType0Font) font;
                    if (type0Font.getToUnicode() == null) {
                        return true;
                    }
                }

                if (considerComplexMath) {
                    if ("Tj".equals(operation) || "'".equals(operation)) {
                        if (isSingleSpecialMathChar(font,
                                ((COSString) arguments.get(0)).getBytes()))
                        {
                            return true;
                        }
                    }

                    if ("TJ".equals(operation)) {
                        COSArray newArray = new COSArray();
                        COSArray array = (COSArray) arguments.get(0);
                        int arraySize = array.size();
                        for (int i = 0; i < arraySize; i++) {
                            COSBase next = array.get(i);
                            if (next instanceof COSString) {
                                if (isSingleSpecialMathChar(font, ((COSString) next).getBytes()))
                                    newArray.add(next);
                            } else {
                                newArray.add(next);
                            }
                        }
                        arguments.clear();
                        arguments.add(newArray);
                        return true;
                    }
                }

                return false;
            }

            return true;
        };
    }

    public static Function3B<PDFStreamEngine, PDFOperator, List<COSBase>> preserveInHtmlHandler(
            final boolean considerComplexMath)
    {
        return (engine, operator, arguments) -> {
            final String operation = operator.getOperation();
            if ("TJ".equals(operation) || "Tj".equals(operation) || "'".equals(operation)
                    || "\"".equals(operation))
            {
                PDFont font = engine.getGraphicsState().getTextState().getFont();

                if (font instanceof PDType0Font) {
                    PDType0Font type0Font = (PDType0Font) font;
                    if (type0Font.getToUnicode() == null)
                        return false;
                }

                if (considerComplexMath) {
                    if ("Tj".equals(operation) || "'".equals(operation)) {
                        if (isSingleSpecialMathChar(font,
                                ((COSString) arguments.get(0)).getBytes()))
                            return false;
                    }

                    if ("TJ".equals(operation)) {
                        COSArray newArray = new COSArray();
                        COSArray array = (COSArray) arguments.get(0);
                        int arraySize = array.size();
                        for (int i = 0; i < arraySize; i++) {
                            COSBase next = array.get(i);
                            if (next instanceof COSString) {
                                if (!isSingleSpecialMathChar(font,
                                        ((COSString) next).getBytes()))
                                    newArray.add(next);
                            } else {
                                newArray.add(next);
                            }
                        }
                        arguments.set(0, newArray);
                        return true;
                    }
                }

                return true;
            }

            return !"Do".equals(operation);
        };
    }

    public static String stripText(PDDocument pdDocument) {
        try {
            StringWriter stringWriter = new StringWriter();
            PDFTextStripper stripper = new PDFTextStripper();

            stripper.writeText(pdDocument, stringWriter);
            return stringWriter.toString();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    public static <T> T withExistingDocument(final InputStreamSource inputStreamSource,
            final boolean force, final Function<PDDocument, T> action)
    {
        return FileUtils.withEmptyTemporaryFile("scratch", ".bin", (IoFunction<File2, T>) scratchFile -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();
            PDDocument document = load(inputStreamSource, scratchFile, force);
            try {
                T result = action.apply(document);
                document.close();
                return result;
            } finally {
                closeQuietly(document);
                stopwatch.stopAndLog("Pdf processing", logger);
            }
        });
    }

    public static void withExistingDocument(final InputStreamSource inputStreamSource,
            final boolean force, final Function1V<PDDocument> action)
    {
        withExistingDocument(inputStreamSource, force, action.asFunctionReturnNull());
    }

    public static void write(final PDDocument document, OutputStreamSource outputStreamSource) {
        FileUtils.withFile(outputStreamSource, file -> {
            try {
                final FileOutputStream fos = new FileOutputStream(file.getFile());
                try {
                    document.setAllSecurityToBeRemoved(true);
                    document.save(fos);
                } finally {
                    IoUtils.closeQuietly(fos);
                }
            } catch (Exception exc) {
                throw ExceptionUtils.translate(exc);
            }
        });
    }

    public static Function1V<PDDocument> writeHandler(OutputStreamSource outputStreamSource) {
        return document -> write(document, outputStreamSource);
    }

    private static DimensionO scale(DimensionO dimension, double multiplier, DoubleUnaryOperator round) {
        return dimension.map(x -> (int) Math.round(round.applyAsDouble(x * multiplier)));
    }

    public static DimensionO getRenderedPageSize(PagesInfo pagesInfo, int zeroBasedPageIndex) {
        ListF<DimensionO> dimensions = pagesInfo.getPageInfos()
                .map(i1 -> new DimensionO(i1.getWidth().map(Math::round), i1.getHeight().map(Math::round)));
        Optional<Integer> maxWidthO = dimensions.stream().map(d -> d.width.get()).max(Integer::compare).filter(w -> w > 1280);
        DimensionO dimension = maxWidthO.map(width -> dimensions.map(d -> scale(d, 1280.0 / width, Math::floor)))
                .orElse(dimensions).get(zeroBasedPageIndex);
        return dimension.width.get() < 794 ? scale(dimension, 794.0 / dimension.width.get(), Math::ceil) : dimension;
    }

}
