package ru.yandex.chemodan.app.docviewer.utils.pdf.image;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.Copier;
import ru.yandex.chemodan.app.docviewer.copy.TempFileInfo;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDao;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.ByteArrayOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.FileList;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.web.backend.StoredResultHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * TODO: merge with pdfImagesHolder or do something
 */
@AllArgsConstructor
public class ImageHelper {

    private static final ListF<String> OLD_MIME_HANDLE =
            Cf.list("mail.yandex.net/dv-mimes", "mail.yandex.ru:443/dv-mimes");
    private static final String NEW_MIME_HANDLE = "meta.mail.yandex.net/mimes";

    private final DynamicProperty<Integer> restoreImageTimeout =
            new DynamicProperty<>("docviewer.image.restore-on-the-fly.timeout[s]", 120);

    public static final Logger logger = LoggerFactory.getLogger(ConvertManager.class);

    private final ExecutorService executorService =
            Executors.newFixedThreadPool(48, new ThreadNameIndexThreadFactory("image-uploader"));
    private final ImageDao imageDao;
    private final FileStorage pdfImagesHolder;
    private final StoredResultHelper storedResultHelper;

    private final Copier copier;
    private final ConvertManager convertManager;
    private final MimeDetector mimeDetector;

    public Option<InputStreamSource> getHtmlBackgroundImageImpl(String fileId, String subKey) {
        Option<StoredImage> one = imageDao.findOne(fileId, subKey);
        if (!one.isPresent()) {
            return Option.empty();
        }
        storedResultHelper.updatePdfImageLastAccess(one.get());
        return Option.of(pdfImagesHolder.get(pdfImagesHolder.toFileLink(one.get().fileLink)));
    }

    public CompletableFuture<Option<FileLink>> addHtmlBackgroundAsync(String fileId, String subKey, File2 file2) {
        return CompletableFuture.supplyAsync(RequestIdStack.withCurrentRequestIdF((Function0<Option<FileLink>>) () -> {
            try {
                FileLink put = pdfImagesHolder.put(file2);
                imageDao.saveImageIfNotExists(fileId, subKey, put.getSerializedPath());
                logger.info("Uploaded image: {} {}", fileId, put);
                return Option.of(put);
            } catch (Exception e) {
                logger.error("Can not uploaded image: {}", fileId);
                return Option.empty();
            } finally {
                file2.deleteRecursiveQuietly();
            }
        })::apply, executorService);
    }

    public Option<InputStreamSource> restoreImage(StoredResult storedResult, String subKey) {
        if (!storedResult.getRestoreUri().isPresent()) {
            logger.info("Stored result for {} doesn't contains restore uri", storedResult.getFileId());
            return Option.empty();
        }

        String restoreUri = storedResult.getRestoreUri().get();

        //hack for CHEMODAN-66493
        if (restoreUri.contains("dv-mimes")) {
            logger.warn("Restore uri of {} {} needs to be fixed",
                    storedResult.getFileId(), storedResult.getConvertTargetType());
            for (String handle : OLD_MIME_HANDLE) {
                restoreUri = restoreUri.replaceFirst(handle, NEW_MIME_HANDLE);
            }
        }

        ActualUri actualUri = new ActualUri(UrlUtils.removeParameter(
                restoreUri, StoredResult.RESTORE_URI_IS_EXTERNAL_FIELD));
        boolean isExternalUri = UrlUtils.getQueryParameterFromUrl(
                restoreUri, StoredResult.RESTORE_URI_IS_EXTERNAL_FIELD).map(Boolean::valueOf).getOrElse(true);

        String session = Random2.R.nextLetters(20);

        TempFileInfo tempFileInfo =
                copier.downloadAndSaveAndExtract(actualUri, isExternalUri, Option.of(session), false);

        Option<String> reportedContentType = tempFileInfo.getReportedContentType();

        String detectedContentType =
                mimeDetector.getMimeType(tempFileInfo.getLocalCopy().getFile(), reportedContentType.unique());

        ListF<File2> images = convertManager.scheduleConvert(
                tempFileInfo.getLocalCopy().getFile(), detectedContentType,
                storedResult.getConvertTargetType(), new ByteArrayOutputStreamSource())
                .getUnchecked(Duration.standardSeconds(restoreImageTimeout.get()))
                .getImages().flatMap(FileList::getFiles);

        Option<FileLink> restored = Option.empty();

        boolean withResultHtml = images.exists(f -> f.getName().equals(ConvertToHtmlHelper.ZIPENTRY_NAME_RESULT_HTML));

        logger.info("Got {} files from converter.{}", images.size(), withResultHtml ? " result.html is present" : "");

        if (images.size() == 1 && !withResultHtml) {
            // case for images from archive
            restored = addHtmlBackgroundAsync(storedResult.getFileId(), subKey, images.single()).join();
        } else if (withResultHtml) {
            // case for excel sheets
            restored = images.filterMap(image -> {
                String imageName = image.getName();
                Option<StoredImage> storedImage = imageDao.findOne(storedResult.getFileId(), imageName);
                Option<FileLink> linkO = storedImage.isPresent()
                                         ? Option.of(pdfImagesHolder.toFileLink(storedImage.get().fileLink))
                                         : addHtmlBackgroundAsync(storedResult.getFileId(), imageName, image).join();

                return imageName.equals(subKey) ? linkO : Option.empty();
            }).firstO();
        }

        images.plus(tempFileInfo.getLocalCopy().getFile()).forEach(file -> {
            try {
                file.safeDelete();
            } catch (Exception e) {
                ExceptionUtils.throwIfUnrecoverable(e);
                logger.error("Failed to delete temp file " + file.getName(), e);
            }
        });

        if (!restored.isPresent()) {
            logger.error("Couldn't restore image for {} {}", storedResult.getFileId(), subKey);
        }

        return restored.map(pdfImagesHolder::get);
    }
}
