package ru.yandex.chemodan.app.docviewer.utils.pdf.image;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupDao;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupTarget;
import ru.yandex.chemodan.app.docviewer.dao.pdfWarmup.PdfWarmupTask;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.Document;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.PdfPageWordsExtractor;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.WordPositionSerializer;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.PrioritizedFutureTask;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.app.docviewer.web.client.DocviewerForwardClient;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author vlsergey
 * @author swined
 */
@AllArgsConstructor
public class PdfHelper {

    private static final Logger logger = LoggerFactory.getLogger(PdfHelper.class);
    private static final String ROLE = PdfHelper.class.getName();

    private final DocviewerForwardClient docviewerForwardClient;
    private final PdfWarmupDao pdfWarmupDao;
    private final Scheduler convertScheduler;
    private final PdfImageCache pdfImageCache;
    private final int prerenderForward;
    private final int prerenderBackward;
    private final Duration maxPdfImageRenderDuration;
    private final Duration getFutureResultTimeout;
    private final FileStorage resultHolder;
    private final StoredResultDao storedResultDao;
    private final TemporaryFileCache temporaryFileCache;
    private final AtomicLong uniqueTaskCounter = new AtomicLong(0);
    private final DynamicProperty<Boolean> fallbackRemoteToLocal =
            new DynamicProperty<>("docviewer.image.fallback-render", false);
//    @Value("${convert.max.pdf.text.page.count:-100}") fix it
    private final int maxPageCount = 100;

    public FileLink getHtmlBackgroundImageLinkInplace(String fileId, int oneBasedPageIndex, DimensionO size, TargetType targetType) {
        FileLink fileLink = pdfImageCache.getFromCache(fileId, oneBasedPageIndex, size)
                .getOrElse(renderLink(fileId, targetType, oneBasedPageIndex, size));
        schedulePrerenderPdfImagesArroundCurrentPage(fileId, oneBasedPageIndex, size, targetType);
        return fileLink;
    }

    public File2 getHtmlBackgroundImageInplace(String fileId, int oneBasedPageIndex, DimensionO size, boolean mobile) {
        TargetType targetType = mobile ? TargetType.HTML_WITH_IMAGES_FOR_MOBILE : TargetType.HTML_WITH_IMAGES;
        return getHtmlBackgroundImageInplace(fileId, oneBasedPageIndex, size, targetType);
    }

    public File2 getHtmlBackgroundImageInplace(String fileId, int oneBasedPageIndex, DimensionO size, TargetType targetType) {
        schedulePrerenderHtmlImageOnWorker(fileId, oneBasedPageIndex, size, targetType);

        Option<FileLink> fromCache = pdfImageCache.getFromCache(fileId, oneBasedPageIndex, size);
        if (fromCache.isPresent()) {
            try {
                return temporaryFileCache.getOrCreateTemporaryFile(fromCache.get(), pdfImageCache.getAsTempFileF());
            } catch (Exception exc) {
                handleCacheError(exc, fileId, oneBasedPageIndex, size);
            }
        }
        FileLink result = renderLink(fileId, targetType, oneBasedPageIndex, size);
        return temporaryFileCache.getOrCreateTemporaryFile(result, pdfImageCache.getAsTempFileF());
    }

    private FileLink renderLink(String fileId, TargetType targetType, int oneBasedPageIndex, DimensionO size) {
        StoredResult storedResult = getStoredResult(fileId, targetType);
        PrioritizedFutureTask<String, FileLink> future = convertScheduler.scheduleLocalTask(
                ROLE + "#" + fileId + "-" + oneBasedPageIndex + "-" + size,
                () -> newHtmlBackgroundTask(storedResult, oneBasedPageIndex, size), 10);

        return future.getUnchecked(maxPdfImageRenderDuration.plus(getFutureResultTimeout));
    }

    private void handleCacheError(Exception exc, String fileId, int oneBasedPageIndex, DimensionO size) {
        logger.error("Error occured on retrieving cached result '" + fileId + "':" + exc, exc);
        try {
            pdfImageCache.removeById(fileId, oneBasedPageIndex, size);
        } catch (Throwable exc2) {
            logger.error("Error occured on removing incorrect cache record '" + fileId + "':" + exc, exc);
        }
    }

    public File2 getHtmlBackgroundImageRemote(String fileId, int oneBasedPageIndex, DimensionO size, boolean mobile) {
        TargetType targetType = mobile ? TargetType.HTML_WITH_IMAGES_FOR_MOBILE : TargetType.HTML_WITH_IMAGES;
        return getHtmlBackgroundImageRemote(fileId, oneBasedPageIndex, size, targetType);
    }

    private File2 getHtmlBackgroundImageRemote(String fileId, int oneBasedPageIndex, DimensionO size, TargetType targetType) {
        try {
            FileLink fileLink = pdfImageCache.getFromCache(fileId, oneBasedPageIndex, size)
                            .getOrElse(() -> forwardToRemote(fileId, size, oneBasedPageIndex, targetType));
            return temporaryFileCache.getOrCreateTemporaryFile(fileLink, pdfImageCache.getAsTempFileF());
        } catch (Exception exc) {
            handleCacheError(exc, fileId, oneBasedPageIndex, size);
            if (fallbackRemoteToLocal.get()) {
                return getHtmlBackgroundImageInplace(fileId, oneBasedPageIndex, size, targetType);
            } else {
                throw exc;
            }
        }
    }

    private FileLink forwardToRemote(String fileId, DimensionO size, int oneBasedPageIndex,
            TargetType targetType)
    {
        FileLink fileLink = resultHolder.toFileLink(docviewerForwardClient
                .forwardToPrerenderSingleHtmlImage(fileId, size, oneBasedPageIndex, isMobile(targetType)));
        logger.info("get FileLink from worker: {} for: {}", fileLink, fileId);
        return fileLink;
    }

    public void removeHtmlBackground(String fileId, int oneBasedPageIndex, DimensionO size) {
        pdfImageCache.removeById(fileId, oneBasedPageIndex, size);
    }

    private <T> T withStoredResult(Duration timeout, StoredResult storedResult, Function<File2, T> func) {
        ThreadLocalTimeout.Handle handle = ThreadLocalTimeout.push(timeout);
        try {
            return func.apply(temporaryFileCache.getOrCreateTemporaryFile(
                    resultHolder.toFileLink(storedResult.getFileLink().get()), resultHolder::getAsTempFile));
        } finally {
            handle.popSafely();
        }
    }

    private FileLink newHtmlBackgroundTask(StoredResult storedResult, int oneBasedPageIndex, DimensionO size) {
        return withStoredResult(maxPdfImageRenderDuration, storedResult,
                temp -> pdfImageCache.putAndGet(storedResult.getFileId(), temp, oneBasedPageIndex, size));
    }

    private StoredResult getStoredResult(String fileId, TargetType targetType) {
        StoredResult storedResult = storedResultDao.find(fileId, targetType).getOrThrow(
                () -> new NotFoundException("File specified were not converted"));
        if (!storedResult.isConvertResultTypePdf()) {
            throw new IllegalArgumentException("Specified file convert result is not PDF type");
        }
        return storedResult;
    }

    private void prerenderPdfImagesAroundCurrentPage(
            String fileId, int oneBasedPageIndex, DimensionO size, TargetType targetType)
    {
        StoredResult storedResult = getStoredResult(fileId, targetType);
        if (storedResult.getPages().isPresent() && (oneBasedPageIndex > storedResult.getPages().get())) {
            logger.info("Wrong page index, oneBasedPageIndex: {}, pages: {}", oneBasedPageIndex, storedResult.getPages());
            return;
        }
        Collection<PdfWarmupTask> tasks = pdfWarmupDao.createTasks(
                new PdfWarmupTarget(
                        fileId,
                        size.width.get(),
                        size.height.get(),
                        targetType == TargetType.HTML_WITH_IMAGES_FOR_MOBILE
                ),
                oneBasedPageIndex - prerenderBackward - 1,
                oneBasedPageIndex + prerenderForward - 1
        );
        if (!tasks.isEmpty()) {
            logger.info("Start to schedule {} tasks for pdf image prerender", tasks.size());

            // Put result in cache before start prerender
            temporaryFileCache.getOrCreateTemporaryFile(
                    resultHolder.toFileLink(storedResult.getFileLink().get()), resultHolder::getAsTempFile);
            for (PdfWarmupTask task : tasks) {
                prescheduleHtmlBackgroundTask(storedResult, task.blockIndex, task.blockSize, size);
            }
        } else {
            logger.info("No need to start pdf image prerender tasks");
        }
    }

    private void newHtmlBackgroundTask(StoredResult storedResult, int blockIndex, int blockSize, DimensionO size) {
        Option<Integer> pages = storedResult.getPages();
        pages.ifPresent(max -> withStoredResult(maxPdfImageRenderDuration.multipliedBy(blockSize), storedResult, temp -> {
            int from = blockIndex * blockSize;
            int to = Math.min(max, from + blockSize);
            for (int i = from + 1; i <= to; i++) {
                pdfImageCache.putAndGet(storedResult.getFileId(), temp, i, size);
            }
            return null;
        }));
    }

    private void prescheduleHtmlBackgroundTask(
            StoredResult storedResult, int blockIndex, int blockSize, DimensionO size)
    {
        String taskSerialized = ROLE + "#prescheduleHtmlBackgroundTask-" + storedResult.getFileId() +
                        "-b" + blockIndex + "*" + blockSize + "-" + size;
        convertScheduler.scheduleLocalTask(taskSerialized, () -> {
            newHtmlBackgroundTask(storedResult, blockIndex, blockSize, size);
            return null;
        }, -1);
    }


    private void schedulePrerenderHtmlImageOnWorker(
            String fileId, int oneBasedPageIndex, DimensionO size, TargetType targetType)
    {
        convertScheduler.scheduleLocalTask(
                ROLE + "#schedulePrerenderHtmlImage-"+ uniqueTaskCounter.getAndIncrement(),
                () -> {
                    docviewerForwardClient.forwardToPrerenderHtmlImage(fileId, size,
                            oneBasedPageIndex, isMobile(targetType));
                    return null;
                },
                -1);
    }

    private boolean isMobile(TargetType targetType) {
        return targetType == TargetType.HTML_WITH_IMAGES_FOR_MOBILE;
    }

    public void schedulePrerenderPdfImagesArroundCurrentPage(String fileId, int oneBasedPageIndex,
            DimensionO size, TargetType targetType)
    {
        convertScheduler.scheduleLocalTask(
                ROLE + "#schedulePreschedule-" + uniqueTaskCounter.getAndIncrement() + "-" + oneBasedPageIndex,
                () -> {
                    prerenderPdfImagesAroundCurrentPage(fileId, oneBasedPageIndex, size, targetType);
                    return null;
                },
                -1);
    }

    public void renderFirstBlock(String fileId, TargetType targetType, PagesInfo pagesInfo) {
        DimensionO size = PdfUtils.getRenderedPageSize(pagesInfo, 0);
        // preschedule normal size (x1) and retina (x2, temporarily disabled)
        schedulePrerenderPdfImagesArroundCurrentPage(fileId, 1, size, targetType);
    }

    public void extractText(String fileId, TargetType targetType, FileLink link) {
        convertScheduler.scheduleLocalTask(
                fileId + "extract-text",
                () -> {
                    try {
                        File2 result = temporaryFileCache.getOrCreateTemporaryFile(
                                link, resultHolder::getAsTempFile);

                        PdfUtils.withExistingDocument(result, true, a -> {
                            Document documentWithExtractedWords = PdfPageWordsExtractor.getDocumentWithExtractedWords(a, Option.empty(),
                                    Option.of(Math.min(a.getNumberOfPages() - 1, maxPageCount)));
                            logger.info("Extracted text for: {}", fileId);
                            FileLink put = resultHolder
                                    .put(new ByteArrayInputStreamSource(WordPositionSerializer.serializeJsonCompressed(documentWithExtractedWords)));
                            storedResultDao.addExtractedText(fileId, targetType, put.getSerializedPath());
                        });

                    } catch (Exception e) {
                        logger.warn("Can't extract text information for {}", fileId, e);
                    }
                    return null;
                },
                1);
    }

}
