package ru.yandex.chemodan.app.docviewer.utils.pdf.image;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.jmx.export.annotation.ManagedOperation;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.ResizeOption;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDao;
import ru.yandex.chemodan.app.docviewer.log.DocviewerTskvEvent;
import ru.yandex.chemodan.app.docviewer.log.DvStages;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.AbstractCache;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfWarmupMetrics;
import ru.yandex.chemodan.app.docviewer.web.backend.StoredResultHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 * @author vlsergey
 * @see ResultsCleanup#cleanupStoredResult(ru.yandex.chemodan.app.docviewer.dao.results.StoredResult)
 */
@AllArgsConstructor
public class PdfImageCache extends AbstractCache {
    private static final Logger logger = LoggerFactory.getLogger(PdfImageCache.class);

    public final DynamicProperty<Integer> pdfimagecacheTtl = new DynamicProperty<>("docviewer.pdfimagecache.ttl", -1);

    private final PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;
    private final FileStorage pdfImagesHolder;
    private final PdfRenderer pdfRenderer;
    private final ImageDao pdfImageDao;
    private final StoredResultHelper storedResultHelper;
    private final TemporaryFileCache temporaryFileCache;

    public Function<FileLink, File2> getAsTempFileF() {
        return pdfImagesHolder::getAsTempFile;
    }

    public Option<FileLink> getFromCache(String fileId, int oneBasedPageIndex, DimensionO size) {
        try {
            PdfWarmupMetrics.totalPages.inc();
            onCacheCall();
            Option<StoredImage> pdfImage = pdfImageDao.findOne(fileId, oneBasedPageIndex, size);

            if (!pdfImage.isPresent()) {
                onCacheMiss();
            } else {
                PdfWarmupMetrics.cachedPages.inc();
                storedResultHelper.updatePdfImageLastAccess(pdfImage.get());
                return Option.of(pdfImagesHolder.toFileLink(pdfImage.get().fileLink));
            }
        } catch (Exception exc) {
            logger.error("Unable to access cache : " + exc, exc);
        }

        return Option.empty();
    }

    public FileLink putAndGet(String fileId, InputStreamSource pdfDocumentSource,
            int oneBasedPageIndex, DimensionO size)
    {
        return DvStages.RENDER.log(
                Tuple2List.fromPairs("fileId", fileId, "oneBasedPageIndex", oneBasedPageIndex, "size", size),
                () -> doPutAndGet(fileId, pdfDocumentSource, oneBasedPageIndex, size)
        );
    }

    private FileLink doPutAndGet(String fileId, InputStreamSource pdfDocumentSource, int oneBasedPageIndex, DimensionO size) {

        Option<FileLink> fromCache = getFromCache(fileId, oneBasedPageIndex, size);
        if (fromCache.isPresent()) {
            return fromCache.get();
        }
        logger.info("Rendering and saving in cache page#{} of PDF {} with dimensions {}",
                oneBasedPageIndex, fileId, size);

        String ext = pdfRenderTargetTypeHolder.getTargetType().value();
        File2 tempImageFile = FileUtils.createEmptyTempFile("pdfimage", "." + ext);
        pdfRenderer.render(pdfDocumentSource, oneBasedPageIndex,
                ResizeOption.scale(size), new FileOutputStreamSource(tempImageFile));
        fromCache = getFromCache(fileId, oneBasedPageIndex, size);
        if (fromCache.isPresent()) {
            temporaryFileCache.getOrCreateTemporaryFile(fromCache.get(), fileLink -> tempImageFile);
            return fromCache.get();
        }

        final FileLink fileLink = pdfImagesHolder.put(tempImageFile);
        storedResultHelper.savePdfImageIfNotExists(fileId, oneBasedPageIndex, size, fileLink.getSerializedPath());
        temporaryFileCache.putInCache(fileLink, tempImageFile);

        return fileLink;
    }

    @ManagedOperation
    public void cleanup(Instant now, Option<Duration> ageO) {
        cleanup(now, ageO.getOrElse(Duration.standardDays(pdfimagecacheTtl.get())));
    }

    @ManagedOperation
    public void removeAll(Instant now) {
        cleanup(now, Duration.ZERO);
    }

    private void cleanup(Instant now, Duration age) {
        Integer ttl = pdfimagecacheTtl.get();
        if (ttl < 0) {
            logger.warn("PDF image cleaning was disabled");
            return;
        }
        pdfImageDao.deleteByLastAccessLessBatch(now.minus(age), this::cleanupObject);
    }

    public void cleanupByFileId(String fileId) {
        pdfImageDao.deleteByFileIdBatch(fileId, this::cleanupObject);
    }

    public void removeById(String fileId, int oneBasedPageIndex, DimensionO size) {
        pdfImageDao.delete(fileId, oneBasedPageIndex, size);
    }

    public void cleanupObject(StoredImage pdfImage) {
        runSafe(() -> pdfImageDao.delete(pdfImage.fileId, pdfImage.subId));
        runSafe(() -> pdfImagesHolder.delete(pdfImagesHolder.toFileLink(pdfImage.fileLink)));
    }


    public void cleanupImageSafeById(String fileId, String subId, Instant lastAccessLimit) {
        Option<StoredImage> imageO = pdfImageDao.findOne(fileId, subId);
        if (!imageO.isPresent()) {
            DocviewerTskvEvent.imageDeleted(fileId, subId).log();
            return;
        }
        StoredImage image = imageO.get();
        if (image.getLastAccess().isAfter(lastAccessLimit)) {
            DocviewerTskvEvent.imageSkipped(fileId, subId, image.getLastAccess()).log();
            return;
        }
        cleanupObject(image);
    }

    private void runSafe(Runnable r) {
        try {
            r.run();
        } catch (Exception e) {
            logger.warn("Couldn't perform operation: {}", e.getMessage());
        }
    }

}
