package ru.yandex.chemodan.app.docviewer.utils.pdf.image;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.PopplerAdapter;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.ResizeOption;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author ssytnik
 */
public class PopplerPdfRenderer implements PdfRenderer {
    private static final Logger logger = LoggerFactory.getLogger(PopplerPdfRenderer.class);

    @Autowired
    private PopplerAdapter popplerAdapter;
    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;
    @Autowired
    private TemporaryFileCache temporaryFileCache;
    @Autowired
    private PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;
    @Autowired
    private ImageMagick imageMagick;

    @Override
    public RenderedImageInfo render(FileLink pdfLink, int pageIndex,
            ResizeOption resize, OutputStreamSource imageFile)
    {
        PdfRenderTargetType targetType = pdfRenderTargetTypeHolder.getTargetType();
        return render(pdfLink, pageIndex, resize, targetType, imageFile);
    }

    @Override
    public RenderedImageInfo render(FileLink pdfLink, int pageIndex, ResizeOption resize,
            PdfRenderTargetType targetType, OutputStreamSource imageFile)
    {
        File2 pdfFile = temporaryFileCache.getOrCreateTemporaryFile(pdfLink, resultHolder::getAsTempFile);
        return render(pdfFile, pageIndex, resize, targetType, imageFile);
    }

    @Override
    public RenderedImageInfo render(InputStreamSource pdfDocumentSource, int pageIndex,
            ResizeOption resize, OutputStreamSource imageFileSource)
    {
        PdfRenderTargetType targetType = pdfRenderTargetTypeHolder.getTargetType();
        return render(pdfDocumentSource, pageIndex, resize, targetType, imageFileSource);
    }

    @Override
    public RenderedImageInfo render(final InputStreamSource pdfDocumentSource, final int pageIndex,
            final ResizeOption resize, final PdfRenderTargetType targetType,
            final OutputStreamSource targetStreamSource)
    {
        return FileUtils.withFile(targetStreamSource, (IoFunction<File2, RenderedImageInfo>) imageFile -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();
            popplerAdapter.convert(pdfDocumentSource, targetStreamSource,
                    pageIndex, resize, targetType, true);
            stopwatch.stopAndLog("Rendering of page " + pageIndex, logger);

            Dimension dim = imageMagick.pingImage(imageFile.getAbsoluteFile()).getOriginalSize();

            return new RenderedImageInfo(Option.of(imageFile.length()),
                    targetType.getMimeType(),
                    dim.getWidth(), dim.getHeight());
        });
    }
}
