package ru.yandex.chemodan.app.docviewer.utils.scheduler;

import java.util.concurrent.Callable;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.thread.InterruptedRuntimeException;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.time.TimeUtils;

@Data
public class PrioritizedFutureTask<K, T> extends FutureTask<T> implements
        Comparable<PrioritizedFutureTask<K, T>>, FutureX<T>
{

    private static <T> Callable<T> inheritLocalTimeout(final Callable<T> callable) {
        final Option<Instant> deadline = ThreadLocalTimeout.deadline();
        if (deadline.isPresent()) {
            return () -> {
                final ThreadLocalTimeout.Handle handle = ThreadLocalTimeout
                        .push(deadline.get());
                try {
                    handle.check();
                    return callable.call();
                } finally {
                    handle.popSafely();
                }
            };
        }
        return callable;
    }

    private static <T> Runnable inheritLocalTimeout(final Runnable runnable) {
        final Option<Instant> deadline = ThreadLocalTimeout.deadline();
        if (deadline.isPresent()) {
            return () -> {
                final ThreadLocalTimeout.Handle handle = ThreadLocalTimeout
                        .push(deadline.get());
                try {
                    handle.check();
                    runnable.run();
                } finally {
                    handle.popSafely();
                }
            };
        }
        return runnable;
    }

    private final Instant created = TimeUtils.now();

    private final K key;

    private volatile float priority;

    public PrioritizedFutureTask(Callable<T> callable, K key, float priority) {
        super(inheritLocalTimeout(callable));
        this.key = key;
        setPriority(priority);
    }

    public PrioritizedFutureTask(Runnable runnable, T result, K key, float priority) {
        super(inheritLocalTimeout(runnable), result);
        this.key = key;
        setPriority(priority);
    }

    @Override
    public int compareTo(PrioritizedFutureTask<K, T> o) {
        if (this.getPriority() == o.getPriority())
            return this.created.compareTo(o.created);

        return 0 - Float.compare(this.getPriority(), o.getPriority());
    }

    public K getKey() {
        return key;
    }

    protected float getPriority() {
        return priority;
    }

    @Override
    public T getUnchecked(Duration timeout) {
        try {
            return get(timeout.getMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException exc) {
            throw new InterruptedRuntimeException(exc);
        } catch (Throwable exc) {
            throw ExceptionUtils.throwException(exc);
        }
    }

    protected void setPriority(final float priority) {
        this.priority = priority;
    }

}
