package ru.yandex.chemodan.app.docviewer.utils.scheduler;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.RunnableFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PrioritizedThreadPoolExecutor<K> extends ThreadPoolExecutor {

    private static final Logger logger = LoggerFactory
            .getLogger(PrioritizedThreadPoolExecutor.class);


    private class PrioritizedFutureTaskImpl<T> extends PrioritizedFutureTask<K, T> {

        PrioritizedFutureTaskImpl(Callable<T> callable, K key, float priority) {
            super(callable, key, priority);
            register(key);
        }

        PrioritizedFutureTaskImpl(Runnable runnable, T result, K key, float priority) {
            super(runnable, result, key, priority);
            register(key);
        }

        @Override
        protected void done() {
            try {
                super.done();
            } finally {
                unregister();
            }
        }

        void register(K key) {
            synchronized (PrioritizedThreadPoolExecutor.this) {
                scheduledOrRunnedTasks.put(key, this);
            }
        }

        @Override
        protected void setException(Throwable t) {
            try {
                super.setException(t);
            } finally {
                unregister();
            }
        }

        void unregister() {
            synchronized (PrioritizedThreadPoolExecutor.this) {
                scheduledOrRunnedTasks.remove(getKey());
            }
        }
    }

    final Map<K, PrioritizedFutureTaskImpl<?>> scheduledOrRunnedTasks = new HashMap<>();

    public PrioritizedThreadPoolExecutor(final String threadNamePrefix, int poolSize,
            final int threadPriority)
    {
        super(poolSize, poolSize, 1, TimeUnit.MINUTES, new PriorityBlockingQueue<>(),
                new ThreadFactory() {

                    final AtomicLong threadsCounter = new AtomicLong(0);

                    @Override
                    public Thread newThread(Runnable r) {
                        final long counter = threadsCounter.incrementAndGet();
                        logger.debug("Creating thread #{} for pool '{}'", counter,
                                threadNamePrefix);
                        Thread thread = new Thread(r, threadNamePrefix + "-" + counter);
                        thread.setPriority(threadPriority);
                        thread.setDaemon(true);
                        return thread;
                    }
                });
        allowCoreThreadTimeOut(true);
    }

    public synchronized boolean isScheduled(K key) {
        return scheduledOrRunnedTasks.containsKey(key);
    }

    @Override
    protected <T> PrioritizedFutureTaskImpl<T> newTaskFor(Callable<T> callable) {
        throw new UnsupportedOperationException("Task key is required");
    }

    protected synchronized <T> PrioritizedFutureTaskImpl<T> newTaskFor(Callable<T> callable, K key,
            float priority)
    {
        return new PrioritizedFutureTaskImpl<>(callable, key, priority);
    }

    @Override
    protected <T> PrioritizedFutureTaskImpl<T> newTaskFor(Runnable runnable, T value) {
        throw new UnsupportedOperationException("Task key is required");
    }

    protected synchronized <T> PrioritizedFutureTaskImpl<T> newTaskFor(
            Runnable runnable, T value, K key, float priority)
    {
        return new PrioritizedFutureTaskImpl<>(runnable, value, key, priority);
    }

    @Deprecated
    @Override
    public <T> Future<T> submit(Callable<T> task) {
        throw new UnsupportedOperationException("Task key is required");
    }

    public synchronized <T> PrioritizedFutureTask<K, T> submit(Callable<T> task, K key, float priority) {
        PrioritizedFutureTask<K, T> scheduled = (PrioritizedFutureTask<K, T>) scheduledOrRunnedTasks.get(key);

        if (scheduled != null) {
            updatePriority(scheduled, priority);
            return scheduled;
        }

        PrioritizedFutureTask<K, T> ftask = newTaskFor(task, key, priority);
        execute(ftask);
        return ftask;
    }

    @Deprecated
    @Override
    public Future<?> submit(Runnable task) {
        throw new UnsupportedOperationException("Task key is required");
    }

    public synchronized Future<?> submit(Runnable task, K key, float priority) {
        PrioritizedFutureTask<K, ?> scheduled = scheduledOrRunnedTasks.get(key);

        if (scheduled != null) {
            updatePriority(scheduled, priority);
            return scheduled;
        }

        RunnableFuture<Object> ftask = newTaskFor(task, null, key, priority);
        execute(ftask);
        return ftask;
    }

    public synchronized boolean updatePriority(PrioritizedFutureTask<K, ?> task, float newPriority) {
        return updatePriorityImpl(task.getKey(), task, newPriority);
    }

    public synchronized boolean updatePriority(K key, float newPriority) {
        PrioritizedFutureTask<K, ?> task = scheduledOrRunnedTasks.get(key);

        if (task == null)
            return false;

        return updatePriorityImpl(key, task, newPriority);
    }

    private boolean updatePriorityImpl(K key, PrioritizedFutureTask<K, ?> task, float newPriority) {
        float oldPriority = task.getPriority();
        if (oldPriority >= newPriority)
            return false;

        boolean removed = getQueue().remove(task);

        if (!removed)
            return false;

        task.setPriority(newPriority);
        execute(task);

        logger.trace("Priority for task with key {} was updated from {} to {}", key, oldPriority, newPriority);

        return true;
    }
}
