package ru.yandex.chemodan.app.docviewer.utils.scheduler.local;

import java.util.concurrent.Callable;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.jmx.export.annotation.ManagedMetric;
import org.springframework.jmx.export.annotation.ManagedResource;
import org.springframework.jmx.support.MetricType;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.docviewer.dao.schedule.ScheduledTask;
import ru.yandex.chemodan.app.docviewer.dao.schedule.ScheduledTaskDao;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.PrioritizedFutureTask;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.PrioritizedThreadPoolExecutor;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.support.tl.ThreadLocalHandle;

/**
 * @author akirakozov
 */
@ManagedResource
public class LocalScheduler implements Scheduler {
    private static final Logger logger = LoggerFactory.getLogger(LocalScheduler.class);

    private final PrioritizedThreadPoolExecutor<String> service;
    private final ScheduledTaskDao scheduledTaskDao;
    private final Duration convertTimeout;

    public LocalScheduler(ScheduledTaskDao scheduledTaskDao,
            PrioritizedThreadPoolExecutor<String> service, Duration convertTimeout)
    {
        this.scheduledTaskDao = scheduledTaskDao;
        this.service = service;
        this.convertTimeout = convertTimeout;
    }

    @Override
    @ManagedMetric(metricType = MetricType.GAUGE)
    public long getQueueLength() {
        return service.getQueue().size();
    }

    @Override
    public int getMaxLocalThreads() {
        return service.getPoolSize();
    }

    @Override
    public boolean isEmpty() {
        return service.getQueue().size() == 0 && service.getActiveCount() == 0;
    }

    @Override
    public boolean isScheduled(String taskKey) {
        if (service.isScheduled(taskKey)) {
            return true;
        }
        Option<ScheduledTask> scheduledTasks = scheduledTaskDao.find(taskKey);
        return scheduledTasks.map(scheduledTask -> {
            Instant plus = scheduledTask.getTimestamp().plus(convertTimeout);
            if (plus.isBefore(Instant.now())) {
                logger.info("Scheduled global task is outdated, removing", taskKey);
                scheduledTaskDao.delete(taskKey);
                return false;
            }
            return true;
        }).getOrElse(false);
    }

    @Override
    public <T> PrioritizedFutureTask<String, T> scheduleGlobalTask(
            final String taskKey, final Callable<T> task, float newPriority)
    {
        logger.info("Schedule global task with key '{}' and priority {}", taskKey, newPriority);
        scheduledTaskDao.saveOrUpdateScheduleItem(taskKey, HostnameUtils.localHostname());
        return submit(taskKey, task, newPriority, () -> scheduledTaskDao.delete(taskKey));
    }

    @Override
    public <T> PrioritizedFutureTask<String, T> scheduleLocalTask(
            final String taskKey, final Callable<T> task, float newPriority)
    {
        logger.info("Schedule local task with key '{}' and priority {}", taskKey, newPriority);
        return submit(taskKey, task, newPriority, Function0V.nop());
    }

    @Override
    @ManagedMetric(metricType = MetricType.GAUGE)
    public int getActiveWorkers() {
        return service.getActiveCount();
    }

    private <T> PrioritizedFutureTask<String, T> submit(
            String taskKey, final Callable<T> task, float newPriority, final Function0V onDoneCallback)
    {
        final String parentRequestId = RequestIdStack.current().getOrElse("");
        final Option<String> ycrid = YandexCloudRequestIdHolder.getO();

        return service.submit(() -> {
            RequestIdStack.Handle handle = RequestIdStack.pushReplace(parentRequestId);
            Option<ThreadLocalHandle> ycridHandle = ycrid.map(YandexCloudRequestIdHolder::setAndPushToNdc);

            try {
                try {
                    return task.call();
                } catch (Exception e) {
                    throw ExceptionUtils.throwException(e);
                } finally {
                    try {
                        onDoneCallback.apply();
                    } catch (Exception e) {
                        logger.warn(e, e);
                    }
                }
            } finally {
                handle.popSafely();
                ycridHandle.ifPresent(ThreadLocalHandle::popSafely);
            }
        }, taskKey, newPriority);
    }
}
