package ru.yandex.chemodan.app.docviewer.web.backend;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ForbiddenException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.InternalServerErrorException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.UnprocessableEntity;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.UnsupportedMediaTypeException;

/**
 * @author metal
 */
public abstract class AbstractActionWithStateValidation<T extends DocviewerRequest> extends AbstractActionServlet<T> {

    @Autowired
    protected StateMachine stateMachine;

    @Autowired
    protected StoredResultDao storedResultDao;

    @Autowired
    private StoredUriDao storedUriDao;

    protected abstract void doGetImpl(HttpServletRequest req, T request, HttpServletResponse resp);

    protected void validateState(State state, ActualUri uri, TargetType targetType, String description) {
        switch (state) {
            case CONVERTING:
            case COPYING:
            case COPIED:
            case NOT_STARTED:
                throw new InternalServerErrorException(
                        "Server did not complete document conversion in allowed time frame (state is '" + state + "')");

            case CONVERTING_ERROR:
                String fileId = stateMachine.getFileId(uri).getOrThrow("File internal id not found");
                ErrorCode convertingErrorErrorCode = storedResultDao.find(fileId, targetType)
                        .getOrThrow("Result not found")
                        .getErrorCode().getOrThrow("Error code not found");
                switch (convertingErrorErrorCode) {
                    case FILE_IS_PASSWORD_PROTECTED:
                    case UNSUPPORTED_PASSWORD_PROTECTED:
                        throw new ForbiddenException("File is password protected");
                    case FILE_IS_EMPTY:
                    case UNSUPPORTED_SOURCE_TYPE:
                    case UNSUPPORTED_CONVERTION:
                        throw new UnsupportedMediaTypeException("Unsupported document for " + description);
                    case TOO_MANY_RETRIES:
                        throw new UnprocessableEntity("Too many retries");
                    default:
                        throw new InternalServerErrorException("Error was occurred during " + description + " " + convertingErrorErrorCode);
                }

            case COPY_ERROR:
                ErrorCode copyErrorErrorCode = storedUriDao.find(uri)
                        .getOrThrow("Uri not found")
                        .getErrorCode().getOrThrow("Error code not found");
                switch (copyErrorErrorCode) {
                    case FILE_NOT_FOUND:
                        throw new NotFoundException("File not found by specified url");
                    case FILE_IS_FORBIDDEN:
                        throw new ForbiddenException("Can't download file because it is forbidden");
                    default:
                        throw new InternalServerErrorException("Error was occurred during document download");
                }

            case NOT_FOUND:
                throw new BadRequestException("Document were not found on remote server");
            case AVAILABLE:
                // ok;
                break;
            default:
                throw new UnsupportedOperationException("Unsupported state: " + state);
        }
    }
}
