package ru.yandex.chemodan.app.docviewer.web.backend;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.Copier;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.MimeHelper;
import ru.yandex.chemodan.app.docviewer.copy.StoredUriManager;
import ru.yandex.chemodan.app.docviewer.copy.TempFileInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.states.MaxFileSizeChecker;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.HttpUtils2;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletResponseOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public abstract class AbstractSourceAction<T extends AbstractSourceRequest> extends AbstractActionServlet<T> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractSourceAction.class);

    @Autowired
    protected StoredUriDao storedUriDao;
    @Autowired
    protected StoredUriManager storedUriManager;
    @Autowired
    protected WebSecurityManager webSecurityManager;
    @Autowired
    protected Copier copier;
    @Value("${mpfs.host}")
    protected String mpfsHost;
    @Autowired
    protected UriHelper uriHelper;
    @Autowired
    protected MimeDetector mimeDetector;
    @Autowired
    protected MimeHelper mimeHelper;
    @Value("${copier.maxFileDownloadingLength}")
    private DataSize fileDownloadingLength;

    private final Function0<SimpleMaxFileSizeChecker> consSimpleMaxFileSizeCheckerF =
            new Function0<SimpleMaxFileSizeChecker>() {
                @Override
                public SimpleMaxFileSizeChecker apply() {
                    return new SimpleMaxFileSizeChecker(fileDownloadingLength);
                }
            }.memoize();


    void extractAndWriteResult(T request, HttpServletRequest req, HttpServletResponse resp) {
        if (request.uid == null) {
            throw new BadRequestException("User ID ('uid') not specified");
        }

        StoredUri storedUri = findFileUri(request);
        ActualUri actualUri = storedUri.getUri();

        Option<String> archivePath = request.getArchivePath();

        Option<FileCopy> temporaryO = Option.empty();
        try {
            ActualUri fileUri = actualUri.withArchivePath(archivePath);

            TempFileInfo tempFileInfo = copier.downloadAndSaveAndExtract(
                    fileUri, uriHelper.isExternalUrl(fileUri), request.getSession(), false,
                    consSimpleMaxFileSizeCheckerF.apply());
            temporaryO = Option.of(tempFileInfo.getLocalCopy());

            boolean isStoredUriOfRequestedFile = actualUri.getArchivePathO().equals(archivePath);
            Option<String> name = archivePath.isPresent() ?
                    Option.empty() : StringUtils.notEmptyO(request.name);
            String filename = getFileName(name, fileUri, tempFileInfo);
            String userAgent = StringUtils.defaultIfEmpty(req.getHeader("User-Agent"), "");

            String respContentType = getContentType(storedUri, tempFileInfo, isStoredUriOfRequestedFile);
            setContentDisposition(resp, storedUri, respContentType, filename, userAgent, request.inline);
            resp.setContentType(respContentType);

            OutputStreamSource outputStreamSource = new ServletResponseOutputStreamSource(resp);
            outputStreamSource.writeFrom(temporaryO.get().getFile());
        } finally {
            temporaryO.ifPresent(fileCopy -> fileCopy.getFile().deleteRecursiveQuietly());
        }
    }

    protected StoredUri findFileUri(T request) {
        if (request.getFileId().isNotEmpty() && request.getUrl().isEmpty()) {
            String fileId = request.getFileId().get();
            webSecurityManager.validateFileRightUsingUid(request.uid, fileId);
            return storedUriManager.findByFileIdAndUidO(fileId, request.uid)
                    .getOrThrow("Could not find stored uri by file id " + fileId + " and uid " + request.uid);
        } else if (request.getUrl().isNotEmpty() && request.getFileId().isEmpty()) {
            DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(request.getUrl().get())
                    .uid(request.uid)
                    .archivePath(request.getArchivePath()).build()
                    .withShowNda(request.showNda);

            ActualUri actualUri = uriHelper.rewrite(source);
            ActualUri actualUriWithoutArchivePath = actualUri.withoutArchivePath();
            webSecurityManager.validateUrlRightUsingUid(request.uid, actualUriWithoutArchivePath);
            return storedUriDao.find(actualUri).orElse(() ->
                    storedUriDao.find(actualUriWithoutArchivePath))
                    .getOrThrow(
                            () -> new NotFoundException("Could not find stored uri by actual uri " + actualUri + " or "
                                    + actualUriWithoutArchivePath));
        } else {
            throw new BadRequestException("File id and url cannot be both empty or both defined");
        }
    }

    private String getFileName(Option<String> name, ActualUri fileUri, TempFileInfo tempFileInfo) {
        return name.getOrElse(tempFileInfo.getContentDispositionFilename().getOrElse(getDefaultFilename(fileUri)));
    }

    private String getDefaultFilename(final ActualUri rewritten) {
        return UriUtils.getFileName(rewritten.getUri().getPath());
    }

    private String getContentType(StoredUri storedUri, TempFileInfo tempFileInfo, boolean isStoredUriOfRequestedFile) {
        return Option.when(isStoredUriOfRequestedFile && StringUtils.isNotEmpty(storedUri.getContentType()),
                storedUri.getContentType())
                .orElse(tempFileInfo.getReportedContentType())
                .orElse(tempFileInfo.getContentDispositionFilename().map(mimeDetector::getMimeTypeByFilename))
                .getOrElse(MimeTypes.MIME_UNKNOWN);
    }

    private void setContentDisposition(HttpServletResponse resp, StoredUri storedUri, String respContentType,
            String filename, String userAgent, boolean inline)
    {
        try {
            String contentDisp = inline
                    && !uriHelper.isExternalUrl(storedUri.getUri())
                    && mimeHelper.isAllowedForInlineDisposition(respContentType)
                    ? "inline" : "attachment";
            String headerValue = contentDisp + "; " + getFilenameMimeParam(filename, userAgent);
            logger.debug("filename: {}, Content-Disposition: {}", filename, headerValue);
            resp.setHeader(HttpUtils2.HEADER_CONTENT_DISPOSITION, headerValue);
        } catch (Exception exc) {
            logger.error("Unable to update result content disposition: " + exc, exc);
        }
    }


    // ssytnik: this is copy/paste from
    // https://svn.yandex.ru/mail/trunk/meta/yandex-disk-downloader/src/DownloaderVars.pm
    static String getFilenameMimeParam(String filename, String userAgent) {
        filename = filename.replace("\'", "\"");
        filename = filename.replaceAll("[\\\\\\/;]", "");
        if (userAgent.contains("MSIE 7.0") || userAgent.contains("MSIE 8.0")) {
            String escaped = percentEncode(filename);
            escaped = escaped.replace("%20", " ");
            escaped = escaped.replace("%25", "%");
            return "filename=" + escaped;
        } else if (userAgent.contains("Safari") && !userAgent.contains("Chrome")) {
            return "filename=" + filename;
        } else {
            return "filename*=UTF-8''" + percentEncode(filename);
        }
    }

    private static String percentEncode(String s) {
        return UrlUtils.urlEncode(s).replace("+", "%20").replace("*", "%2A");
    }

    private static class SimpleMaxFileSizeChecker implements MaxFileSizeChecker {
        private final DataSize limit;

        SimpleMaxFileSizeChecker(DataSize limit) {
            this.limit = limit;
        }

        @Override
        public DataSize getMaxFileLength(String mimeType) {
            return limit;
        }

        @Override
        public DataSize getMaxArchiveLength() {
            return limit;
        }
    }
}
