package ru.yandex.chemodan.app.docviewer.web.backend;

import org.dom4j.Document;
import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author ssytnik
 */
@SuppressWarnings("serial")
public class ArchiveListingAction extends AbstractXmlActionServlet<FileRequest> implements BackendServlet {
    private static final Logger logger = LoggerFactory.getLogger(ArchiveListingAction.class);

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;
    @Autowired
    private StoredResultHelper storedResultHelper;
    @Autowired
    private TemporaryFileCache temporaryFileCache;
    @Autowired
    WebSecurityManager webSecurityManager;
    @Autowired
    private StoredResultDao storedResultDao;

    @Override
    public String getActionUrl() {
        return "/archive-listing";
    }

    @Override
    public void execute(FileRequest request, Document doc) {
        if (request.uid == null) {
            throw new BadRequestException("User ID ('uid') not specified");
        }
        if (StringUtils.isEmpty(request.id)) {
            throw new BadRequestException("No file id specified in request");
        }

        webSecurityManager.validateFileRightUsingUid(request.uid, request.id);

        // XXX proper target type, either accept from frontend, or add special TargetType.ARCHIVE_LISTING
        StoredResult storedResult = storedResultHelper.getValidatedResult(
                request.id, getConvertTargetMobileIncluded(request));

        if (!storedResult.getConvertResultType().isSome(ConvertResultType.ARCHIVE_LISTING)) {
            throw new BadRequestException("Specified file is not an archive");
        }

        final File2 temporary = getResultFile(storedResult, request);

        Document listingDoc = Dom4jUtils.read(temporary);
        doc.add(listingDoc.getRootElement().detach());
    }

    private File2 getResultFile(StoredResult storedResult, DocviewerRequest request) {
        final FileLink resultFileLink = resultHolder.toFileLink(storedResult.getFileLink().get());
        try {
            return temporaryFileCache.getOrCreateTemporaryFile(resultFileLink, resultHolder::getAsTempFile);
        } catch (HttpException e) {
            // Temporary hack to fix inconsistent data
            // https://st.yandex-team.ru/DOCVIEWER-1969
            if (e.getStatusCode().isSome(HttpStatus.NOT_FOUND_404)) {
                logger.debug("Remove result meta, because couldn't find result in storage: " + resultFileLink);
                storedResultDao.delete(storedResult.getFileId(), getConvertTargetMobileIncluded(request));
            }
            throw e;
        }
    }

}
