package ru.yandex.chemodan.app.docviewer.web.backend;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.openoffice.ExtendedDocumentFormatRegistry;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.states.StartManager;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeMalformedURLException;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeURISyntaxException;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.OutputStreamOutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author metal
 */
public class ConvertToMsOoxmlAction extends AbstractActionWithStateValidation<ConvertToMsOoxmlRequest> implements BackendServlet {
    private static final Logger logger = LoggerFactory.getLogger(ConvertToMsOoxmlAction.class);

    @Value("${document.convert.ooxml.office.timeout}")
    private Duration convertTimeout;

    @Autowired
    private StartManager startManager;

    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    private FileStorage resultHolder;
    @Autowired
    private TemporaryFileCache temporaryFileCache;

    @Override
    public String getActionUrl() {
        return "/convert-to-ms-ooxml";
    }

    @Override
    protected void doGetImpl(HttpServletRequest req, final ConvertToMsOoxmlRequest request,
            final HttpServletResponse resp)
    {
        DocumentSourceInfo source = DocumentSourceInfo.builder()
                .originalUrl(request.url)
                .uid(request.uid)
                .archivePath(StringUtils.notEmptyO(request.archivePath)).build()
                .withShowNda(request.showNda);

        ActualUri actualUri = uriHelper.rewrite(source);

        State state = convertDocument(request, source);
        logger.debug("'{}' processing completed with state '{}'", actualUri, state);
        validateState(state, actualUri, TargetType.MICROSOFT_OOXML_OFFICE, "converting to Microsoft OOXML office format");

        String fileId = stateMachine.getFileId(actualUri).getOrThrow("File internal id not found");
        Option<StoredResult> storedResultO = storedResultDao.find(fileId, TargetType.MICROSOFT_OOXML_OFFICE);
        if (storedResultO.isEmpty()) {
            throw new IllegalStateException("Stored result must be available");
        }

        writeConvertedFile(resp, storedResultO.get());
    }

    private State convertDocument(ConvertToMsOoxmlRequest request, DocumentSourceInfo source) throws BadRequestException {
        try {
            if (!request.unsafe) {
                webSecurityManager.validateUrl(source.getOriginalUrl(), source.getUid());
            }
            return startManager.startAndWaitUntilComplete(
                    source,
                    StringUtils.notEmptyO(request.contentType),
                    TargetType.MICROSOFT_OOXML_OFFICE, convertTimeout, true);
        } catch (RuntimeURISyntaxException | RuntimeMalformedURLException exc) {
            throw new BadRequestException("Specified URL '" + request.url + "' is incorrect");
        }
    }



    private void writeConvertedFile(final HttpServletResponse resp, StoredResult storedResult) {
        final FileLink fileLink = resultHolder.toFileLink(storedResult.getFileLink().get());
        final File2 resultFile = temporaryFileCache.getOrCreateTemporaryFile(fileLink, resultHolder::getAsTempFile);

        String extension = storedResult.getConvertResultType().get().getExtension();
        String mimeType = ExtendedDocumentFormatRegistry.INSTANCE.getFormatByFileExtension(extension).getMimeType();

        try {
            resp.setStatus(HttpStatus.SC_200_OK);
            resp.setContentType(mimeType);
            resp.setContentLength((int) resultFile.length());

            new OutputStreamOutputStreamSource(resp.getOutputStream()).writeFrom(resultFile);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
